function add(obj, varargin)
%ADD Add entry to IVI conifiguration store object.
%
%   ADD(OBJ, 'TYPE', 'NAME', ...) adds a new entry TYPE to the IVI
%   configuration store object, OBJ with name, NAME. If an entry of
%   TYPE with NAME exists an error will occur. Based on TYPE additional
%   arguments are required. TYPE can be HardwareAsset, DriverSession or
%   LogicalName.
%
%   ADD(OBJ, 'DriverSession', 'NAME', 'MODULENAME', 'HARDWAREASSETNAME',
%   'P1', 'V1',...) adds a new driver session entry to the IVI
%   configuration  store object, OBJ with name, NAME, using the specified
%   software module name, MODULENAME and hardware asset name,
%   HARDWAREASSETNAME. Optional param-value pairs may be included. All
%   on/off properties are assumed if not specified. Valid parameters for a
%   DriverSession  are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Description           any string    Description of driver session
%     VirtualNames          struct        A struct array containing virtual
%                                         name mappings.
%     Cache                 on/off        Enable caching if the driver supports it.
%     DriverSetup           any string    This value is software module dependent.
%     InterchangeCheck      on/off        Enable driver interchangeability
%                                         checking if supported.
%     QueryInstrStatus      on/off        Enable instrument status querying
%                                         by the driver.
%     RangeCheck            on/off        Enable extended range checking by
%                                         the driver if supported.
%     RecordCoercions       on/off        Enable recording of coercions by
%                                         the driver if supported.
%     Simulate              on/off        Enable simulation by the driver.
%
%   ADD(OBJ, 'HardwareAsset', 'NAME', 'IORESOURCE', 'P1', 'V1') adds a new
%   hardware asset entry to the IVI configuration store object, OBJ, with
%   name, NAME and resource descriptor, IORESOURCE. Optional param-value
%   pairs may be included. Valid parameters for a HardwareAsset are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Description           any string    Description of hardware asset
%
%   ADD(OBJ, 'LogicalName', 'NAME', 'SESSIONNAME', 'P1', 'V1') adds a new
%   logical name entry to the IVI configuration store object, OBJ, with
%   name, NAME and driver session name, SESSIONNAME. Optional param-value
%   pairs may be included. Valid parameters for logical names are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Description           any string    Description of logical name
%
%   ADD(OBJ, S) where S is a structure whose field names are the entry
%   parameter names, adds an entry to IVI configuration store object, OBJ,
%   of the specified type with the values contained in the structure. S 
%   must have a Type field which can be DriverSession, HardwareAsset or
%   LogicalName. S must also have a Name field which is the name of the
%   driver session, hardware asset or logical name to be added.
%
%   Additions made to the configuration store object, OBJ, can be saved
%   to the configuration store data file with the COMMIT function.
%
%   Examples:
%       % Construct IVI configuration store object, c.
%       c = iviconfigurationstore;
%
%       % Add a hardware asset with name, gpib1, and resource description
%       % GPIB0::1::INSTR.
%       add(c, 'HardwareAsset', 'gpib1', 'GPIB0::1::INSTR');
%
%       % Add a driver session with name, S1 that uses the TekScope software
%       % module and the hardware asset with name gpib1.
%       add(c, 'DriverSession', 'S1', 'TekScope', 'gpib1');
%
%       % Add a logical name to configuration store object, c, with name,
%       % MyScope, driver session name, S1 and description, A logical name.
%       add(c, 'LogicalName','MyScope', 'S1', 'Description', 'A logical name');
%
%       % Add a hardware asset with name, gpib3, and resource description,
%       % GPIB0::3::ISNTR.
%       s.Name = 'gpib3';
%       s.IOResourceDescriptor = 'GPIB0::3::INSTR'
%       add(c, s);
%
%       % Save changes to the IVI configuration store data file.
%       commit(c);
%
%   See also IVICONFIGURATIONSTORE/UPDATE, IVICONFIGURATIONSTORE/REMOVE,
%   IVICONFIGURATIONSTORE/COMMIT.
%

%   PE 10-01-03
%   Copyright 1999-2004 The MathWorks, Inc.
%   $Revision: 1.1.6.4 $  $Date: 2004/12/01 20:12:06 $

error(nargchk(2, inf, nargin));

arg = varargin{1};

if isstruct(arg)
    
    % Handle new item(s) based in a struct (array) argument.
    
    if (nargin > 2)
        error('iviconfigurationstore:add:tooManyArgs', 'Too many input arguments.');
    end
    
    for idx = 1:length(arg)
        try
            localAddStructEntry(obj.cobject, arg(idx));
        catch
            rethrow(lasterror);
        end
    end
elseif ischar(arg)
    
    % Handle new item based on field/value pairs.
    
    error(nargchk(4, inf, nargin));

    if (~privateVerifyIsMutableEntry(arg))
        error('iviconfigurationstore:add:invalidArgs', 'Invalid TYPE.');
    end

    if (~ischar(varargin{2}))
        error('iviconfigurationstore:add:invalidArgs', 'NAME must be a string.');
    end

    try
        localAddPairedEntry(obj.cobject, varargin{:})
    catch
        rethrow(lasterror);
    end
else
    error('iviconfigurationstore:add:invalidArgs', 'Invalid TYPE or STRUCT.');
end

% ------------------------------------------------------------------------
% Add a new entry with a struct argument.
% ------------------------------------------------------------------------
function localAddStructEntry(comObj, mStructArray)
     
try
    for idx = 1:length(mStructArray)

        mStruct = mStructArray(idx);

        if isfield(mStruct, 'Type') && isfield(mStruct, 'Name')
            
            if (~privateVerifyIsMutableEntry(mStruct.Type))
                error('iviconfigurationstore:add:invalidArgs', 'Invalid Type field in STRUCT.');
            end

            localAddEntry(comObj, mStruct);
        else
            error('iviconfigurationstore:add:invalidArgs',...
                'Invalid STRUCT specified. STRUCT must have ''Type'' and ''Name'' fields.');
        end

    end
catch
    rethrow(lasterror);
end

% ------------------------------------------------------------------------
% Add a new entry with field value pair arguments.
% ------------------------------------------------------------------------
function localAddPairedEntry(comObj, varargin)

mStruct = privateCreateStruct(varargin{1}, varargin{2});

switch mStruct.Type
    case 'HardwareAsset'
        mStruct.IOResourceDescriptor = varargin{3};
        mStruct = privateFillStruct(mStruct, varargin{4:end});
        localAddEntry(comObj, mStruct);
    case 'DriverSession'
        error(nargchk(5, inf, nargin));

        mStruct.SoftwareModule = varargin{3};
        mStruct.HardwareAsset = varargin{4};

        mStruct = privateFillStruct(mStruct, varargin{5:end});
        localAddEntry(comObj, mStruct);
    case 'LogicalName'
        mStruct.Session = varargin{3};
        mStruct = privateFillStruct(mStruct, varargin{4:end});
        localAddEntry(comObj, mStruct);
end

% ------------------------------------------------------------------------
% Add a new entry to the appropriate configuration store collection.
% ------------------------------------------------------------------------
function localAddEntry(comObj, mStruct)

% Verify that an existing entry with this name does not exist.

collection = comObj.([mStruct.Type 's']);

if isempty(collection) || isstruct(collection)
    return;
end

comEntry = privateGetNamedEntry(collection, mStruct.Name);

if (~isempty(comEntry))
    error('iviconfigurationstore:add:invalidEntry', 'An entry with the same name exists. Use update to alter an existing entry.');
end

% Create a new blank entry.

comEntry = actxserver(['IviConfigServer.Ivi' mStruct.Type]);

% Update the entry values based on the values provided by the user.

privateUpdateEntry(comEntry, comObj, mStruct);

% Add the new entry to the collection.

collection.Add(comEntry);
