function remove(obj, varargin)
%REMOVE Remove entry from IVI configuration store object.
%
%   REMOVE(OBJ, 'TYPE', 'NAME') removes an entry of type, TYPE, with name,
%   NAME, from the IVI configuration store object, OBJ. TYPE can be
%   HardwareAsset, DriverSession or LogicalName.
%
%   REMOVE(OBJ, STRUCT) remove an entry using the fields in STRUCT.
%
%   If a HardwareAsset is removed that is referenced by existing DriverSessions
%   in the configuration store, that reference will be removed from those
%   DriverSessions.
%
%   If a DriverSession is removed that is reference by existing LogicalNames in
%   the configuration store, that reference will be removed from those
%   DriverSessions.
%
%   Examples:
%       c = iviconfigurationstore;
%       remove(c, 'HardwareAsset', 'gpib1');
%
%   See also IVICONFIGURATIONSTORE/UPDATE, IVICONFIGURATIONSTORE/ADD,
%   IVICONFIGURATIONSTORE/COMMIT.

%   PE 10-01-03
%   Copyright 1999-2004 The MathWorks, Inc.
%   $Revision: 1.1.6.4 $  $Date: 2004/12/01 20:12:17 $

nargchk(2, 3, nargin);

arg = varargin{1};

if isstruct(arg)
    
    % Handle remove based on struct (array).
    
    for idx = 1:length(arg)
        
        item = arg(idx);
        
        if isfield(item, 'Type') && isfield(item, 'Name')
            if (~privateVerifyIsMutableEntry(item.Type))
                error('iviconfigurationstore:remove:invalidArgs', 'Invalid Type field in STRUCT.');
            end

            try
                localRemoveEntry(obj.cobject, item.Type, item.Name);
            catch
                rethrow(lasterror);
            end
        else
            warning('iviconfigurationstore:remove:invalidArgs',...
                'Invalid STRUCT ignored. STRUCT must have ''Type'' and ''Name'' fields.');
        end
        
    end
elseif (ischar(arg))
    
    % Handle remove based on name and type.
    
    if (~privateVerifyIsMutableEntry(arg))
        error('iviconfigurationstore:remove:invalidArgs', 'Invalid TYPE.');
    end

    if (nargin < 3 || ~ischar(varargin{2}))
        error('iviconfigurationstore:remove:invalidArgs', 'NAME must be a string.');
    end

    try
        localRemoveEntry(obj.cobject, arg, varargin{2});
    catch
        rethrow(lasterror);
    end
else
    error('iviconfigurationstore:remove:invalidArgs', 'Invalid TYPE or STRUCT.');
end

% ------------------------------------------------------------------------------
% Remove the entry from the collection.
% ------------------------------------------------------------------------------
function localRemoveEntry(comObj, type, name)

collection = comObj.([type 's']);

if isempty(collection) || isstruct(collection)
    return;
end

[unused, index] = privateGetNamedEntry(collection, name);

if isempty(index)
    warning('iviconfigurationstore:remove:notFound', [name ' is not a valid ' type ' entry and was not removed.']);
end

% When removing a HardwareAsset, any DriverSessions referencing that asset must
% be updated.
% When removing a DriverSession, any LogicalNames referencing that asset must be
% updated.

switch (lower(type))
    case 'hardwareasset'
        localDereferenceDriverSessions(comObj, name);
    case 'driversession'
        localDereferenceLogicalNames(comObj, name);
end

try
    collection.Remove(index);
catch
    warning('iviconfigurationstore:remove:invalidArgs', [name ' could not be removed.  It may be referenced by another entry in the configuration store.']);
end

% ------------------------------------------------------------------------------
% Update any logical names referencing the DriverSession to be removed.
% ------------------------------------------------------------------------------
function localDereferenceLogicalNames(comObj, sessionName)

collection = comObj.LogicalNames;

if (isempty(collection))
    return;
end

for idx = 1:collection.Count
    if (~isempty(collection.Item(idx).Session))
        if (strcmp(collection.Item(idx).Session.Name, sessionName) == 1)
            collection.Item(idx).Session = [];
        end
    end
end

% ------------------------------------------------------------------------------
% Update any driver sessions referencing the HardwareAsset to be removed.
% ------------------------------------------------------------------------------
function localDereferenceDriverSessions(comObj, assetName)

collection = comObj.DriverSessions;

if (isempty(collection))
    return;
end

for idx = 1:collection.Count
    if (~isempty(collection.Item(idx).HardwareAsset))
        if (strcmp(collection.Item(idx).HardwareAsset.Name, assetName) == 1)
            collection.Item(idx).HardwareAsset = [];
        end
    end
end
