function update(obj, varargin)
%UPDATE Update entry in IVI configuration store object.
%
%   UPDATE(OBJ, 'TYPE', 'NAME', P1, V1, ...) updates an entry of type, 
%   TYPE, with name, NAME, in IVI configuration store object, OBJ, using 
%   the specified param-value pairs. TYPE can be HardwareAsset,
%   DriverSession, or LogicalName.
%
%   If an entry of TYPE with NAME does not exist, an error will occur.
%
%   Valid parameters for a DriverSession are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Name                  string        A unique name for the driver session.
%     SoftwareModule        string        The name of a software module entry in the configuration store.
%     HardwareAsset         string        The name of a hardware asset entry in the configuration store.
%     VirtualNames          struct        A struct array containing virtual name mappings.
%     Description           any string    Description of driver session
%     Cache                 on/off        Enable caching if the driver supports it.
%     DriverSetup           any string    This value is software module dependent.
%     InterchangeCheck      on/off        Enable driver interchangeability checking if supported.
%     QueryInstrStatus      on/off        Enable instrument status querying by the driver.
%     RangeCheck            on/off        Enable extended range checking by the driver if supported.
%     RecordCoercions       on/off        Enable recording of coercions by the driver if supported.
%     Simulate              on/off        Enable simulation by the driver.
%
%   Valid parameters for a HardwareAsset are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Name                  string        A unique name for the hardware asset.
%     Description           any string    Description of the hardware asset.
%     IOResourceDescriptor  string        The I/O address of the hardware asset.
%
%   Valid parameters for a LogicalName are:
%
%     Param Name            Value         Description
%     ----------            ------        ------------
%     Name                  string        A unique name for the logical name.
%     Description           any string    Description of hardware asset
%     Session               string        The name of a driver session entry in the configuration store.
%
%   UPDATE(OBJ, STRUCT) updates the entry using the fields in STRUCT. If an
%   entry with the type and name field in STRUCT does not exist, an error
%   will occur. Note that the Name field cannot be updated using this syntax.
%
%   Examples:
%       c = iviconfigurationstore;
%       update(c, 'DriverSession', 'ScopeSession', 'Description', 'A session.');
%
%   See also IVICONFIGURATIONSTORE/ADD, IVICONFIGURATIONSTORE/REMOVE,
%   IVICONFIGURATIONSTORE/COMMIT.
%

%   PE 10-01-03
%   Copyright 1999-2004 The MathWorks, Inc. 
%   $Revision: 1.1.6.6 $  $Date: 2004/12/01 20:12:18 $

error(nargchk(2, inf, nargin));

arg = varargin{1};

if isstruct(arg)

    % Handle update based in a struct (array) argument.
    
    if (nargin > 2)
        error('iviconfigurationstore:update:tooManyArgs', 'Too many input arguments.');
    end
    
    for idx = 1:length(arg)
        
        mStruct = arg(idx);
        
        if isfield(mStruct, 'Type') && isfield(mStruct, 'Name')
            if (~privateVerifyIsMutableEntry(mStruct.Type))
                error('iviconfigurationstore:update:invalidArgs', 'Invalid Type field in STRUCT.');
            end
            
            try
                localUpdateEntry(obj.cobject, mStruct.Name, mStruct);
            catch
                rethrow(lasterror);
            end
        else
            warning('iviconfigurationstore:update:invalidArgs',...
                'Invalid STRUCT ignored. STRUCT must have ''Type'' and ''Name'' fields.');
        end
    end
elseif ischar(arg)
    
    % Handle update based on field/value pairs.

    error(nargchk(4, inf, nargin));

    if (~privateVerifyIsMutableEntry(arg))
        error('iviconfigurationstore:update:invalidArgs', 'Invalid TYPE.');
    end

    if (~ischar(varargin{2}))
        error('iviconfigurationstore:update:invalidArgs', 'NAME must be a string.');
    end
    
    try
        entry.Type = arg;
        entry = privateFillStruct(entry, varargin{3:end});
        localUpdateEntry(obj.cobject, varargin{2}, entry);
    catch
        rethrow(lasterror);
    end

else
    error('iviconfigurationstore:update:invalidArgs', 'Invalid TYPE or STRUCT.');
end


% -------------------------------------------------------------------------
% Update the configuration store entry.
% -------------------------------------------------------------------------
function errflag = localUpdateEntry(comObj, name, mStruct)

errflag = false;

collection = comObj.([mStruct.Type 's']);

if isempty(collection) || isstruct(collection)
    return;
end

comEntry = privateGetNamedEntry(collection, name);

if (isempty(comEntry))
    error('iviconfigurationstore:update:invalidArgs', [name ' is not a valid ' mStruct.Type ' entry.']);
end

privateUpdateEntry(comEntry, comObj, mStruct);
