function varargout = makemid(driver, varargin)
% MAKEMID Convert a driver to MATLAB instrument driver format.
%
%   MAKEMID('DRIVER') searches through known driver types for DRIVER and creates
%   a MATLAB instrument driver representation of the driver. Known driver types
%   include VXIplug&play, IVI-C and IVI-COM. The MATLAB instrument driver will
%   be saved in  the current working directory as DRIVER.MDD
%
%   The MATLAB instrument driver can then be modified using MIDEDIT to customize
%   the  driver behavior, and may be used to instantiate a device object using
%   ICDEVICE.
%
%   MAKEMID('DRIVER', 'FILENAME') saves the newly created MATLAB instrument
%   driver using the name and path specified by FILENAME.
% 
%   MAKEMID('DRIVER', 'TYPE') and MAKEMID('DRIVER', 'FILENAME', 'TYPE') override
%   the default search order and only look for drivers of type TYPE. Valid
%   types are vxiplug&play, ivi-c, and ivi-com.
%
%   MAKEMID('DRIVER', 'TYPE', 'INTERFACE') and MAKEMID('DRIVER', 'FILENAME',
%   'TYPE', 'INTERFACE') specifies the IVI-COM driver interface to be used for
%   the object. TYPE must be ivi-com'. The driver root interface is serched for
%   the named interface. For example, if the driver supports the IIviScope
%   interface, an INTERFACE value 'IIviScope' will result in a device object
%   that only contains the IVIScope class-compliant properties and methods.
%
%    Examples:
%        makemid('hp34401');
%        makemid('tktds5k', 'C:\MyDrivers\tektronix_5k.mdd');
%        makemid('tktds5k', 'ivi-c');
%        makemid('MyIviLogicalName');
%
%    See also ICDEVICE, MIDEDIT.

%   PE 12-01-03   
%   Copyright 1999-2004 The MathWorks, Inc. 
%   $Revision: 1.1.6.6 $  $Date: 2004/12/01 20:12:39 $

errorID = 'instrument:makemid:conversionError';

error(nargchk(1, 4, nargin, 'struct'))

if (~ischar(driver))
    error(errorID, 'DRIVER must be a string. Type ''help makemid'' for valid arguments.');
end

interface  = '';
drivertype = '';

switch nargin
    case 1      
        filename = localValidateExtension(fullfile(pwd, driver));
    case 2
        if (~ischar(varargin{1}))
            error(errorID, 'TYPE must be a string. Type ''help makemid'' for valid arguments.');
        end
        
        if (localIsDriverType(varargin{1}))
            filename = localValidateExtension(fullfile(pwd, driver));
            drivertype = varargin{1};
        else
            filename = localValidateExtension(varargin{1});
        end
    case 3
        if (~ischar(varargin{1}))
            error(errorID, 'TYPE or FILENAME must be a string. Type ''help makemid'' for valid arguments.');
        end
        
        if (~ischar(varargin{2}))
            error(errorID, 'TYPE or INTERFACE must be a string. Type ''help makemid'' for valid arguments.');
        end
        
        if (localIsDriverType(varargin{2}))
            filename = localValidateExtension(varargin{1});
            drivertype = varargin{2};
        else
            drivertype = varargin{1};
            interface = varargin{2};
            filename = localValidateExtension(fullfile(pwd, driver));
        end
    case 4
        if (~ischar(varargin{1}))
            error(errorID, 'FILENAME must be a string. Type ''help makemid'' for valid arguments.');
        end
        
        if (~ischar(varargin{3}))
            error(errorID, 'INTERFACE must be a string. Type ''help makemid'' for valid arguments.');
        end
        
        filename = localValidateExtension(varargin{1});
        drivertype = varargin{2};
        interface = varargin{3};
end

% Verify the specified path is not a directory.

if (exist(filename, 'dir'))
    error(errorID, 'FILENAME can not be a directory. Type ''help makemid'' for valid arguments.');
end

% Verify the interface name is only defined for IVI-COM drivers.

if (~isempty(interface) && ~isempty(drivertype) && ~strcmpi(drivertype, 'ivi-com'))
     error(errorID, 'INTERFACE can only be specified when TYPE is ivi-com. Type ''help makemid'' for valid arguments.');
end

model = [];

% Attempt to create a DriverModel based on the specified driver type, or the
% driver that is found.

try
    if (~isempty(drivertype))
        switch lower(drivertype)
            case 'vxiplug&play'
                model = localConvertFromVxi(driver);
            case 'ivi-c'
                model = localConvertFromIviC(driver);
            case 'ivi-com'
                model = localConvertFromIviCom(driver, interface);
        end
    else
        model = localConvertFromIviCom(driver, interface);

        if isempty(model)
            model = localConvertFromIviC(driver);
        end

        if isempty(model)
            model = localConvertFromVxi(driver);
        end
    end
catch
    error(errorID, lasterr);
end

if (isempty(model))
    error(errorID, 'The specified driver could not be loaded.');
end

% Attempt to write the driver data.

try
    writer = com.mathworks.toolbox.instrument.device.guiutil.midtool.DriverFileWriter(model);

    % Throws TMException with appropriate error if it fails.
    writer.writeXML(filename, pwd);
catch
    % Better error formatting if we don't simply allow the exception through.
    error(errorID, lasterr);
end

if nargout
    varargout{1} = model;
end

%-------------------------------------------------------------------------------
% Enforce the .mdd extension if necessary.  Append if no extension.
%
function str = localValidateExtension(str)

if (length(str) < 5)
    return;
end

idx = strfind(str, '.');

if (~isempty(idx))
    str(idx(end):end) = '';
end

str = [str '.mdd'];

%-------------------------------------------------------------------------------
%
function isdrivertype = localIsDriverType(str)

isdrivertype =  any(strcmpi(str, {'vxiplug&play','ivi-c','ivi-com'}));

%-------------------------------------------------------------------------------
%
function model = localConvertFromVxi(driver)
import com.mathworks.toolbox.instrument.device.drivers.vxipnp.VXIPnPLoader;

model = [];

prefix = instrgate('privateGetVXIPNPPath');

if (isempty(prefix))
    return
end;

frontPanelFile = fullfile(prefix, driver, [driver '.fp']);

if (exist(frontPanelFile, 'file'))
    model = VXIPnPLoader.toDriverModel(frontPanelFile, false);
end

%-------------------------------------------------------------------------------
%
function model = localConvertFromIviC(driver)

import com.mathworks.toolbox.instrument.device.drivers.vxipnp.VXIPnPLoader;

model = [];

prefix = instrgate('privateGetIviPath');

if (isempty(prefix))
    return
end;

frontPanelFile = fullfile(prefix, 'Drivers', driver, [driver '.fp']);

if (exist(frontPanelFile, 'file'))
    model = VXIPnPLoader.toDriverModel(frontPanelFile, true);
end

%-------------------------------------------------------------------------------
%
function model = localConvertFromIviCom(driver, interface)

[unused, model] = instrgate('privateDeviceConstructorHelper', ...
    'iviParserModelFromLogical', driver, interface);

if (isempty(model))
    [unused, model] = instrgate('privateDeviceConstructorHelper', ...
        'iviParserModelFromProgramId', driver, interface);
end
