%% VISA Binary Read and Write Operations
%
% Copyright 1999-2004 The MathWorks, Inc. 
% $Revision: 1.7.2.6 $  $Date: 2004/03/24 20:40:29 $

%% Introduction
% This demo explores binary read and write operations with a 
% VISA object.
% 
% The information obtained for this demonstration was prerecorded.
% Therefore, you do not need an actual instrument  to learn about binary
% read and write operations using a  VISA object.
% 
% The GPIB board used was a National Instruments PCI-GPIB+ GPIB card. The
% instrument used was a Tektronix TDS 210 oscilloscope.

%% The Supported VISA Interfaces
% The VISA object supports seven interfaces:
%
% * serial
% * GPIB
% * VXI
% * GPIB-VXI
% * TCPIP
% * USB
% * RSIB
% 
% This tutorial explores binary read and write operations
% using a VISA-GPIB object. However, binary read and write 
% operations for VISA-GPIB, VISA-VXI, VISA-GPIB-VXI, VISA-TCPIP,
% and VISA-USB objects are identical to each other. 
% Therefore, you can use the same commands. The only difference
% is the resource name specified in the VISA constructor.
% 
% Binary read and write operations for the VISA-serial object
% are identical to binary read and write operations for the
% serial port object. Therefore, to learn how to perform binary
% read and write operations for the VISA-serial object, you
% should refer to the Serial Port Binary Read/Write tutorial.
% 
% Binary read and write operations for the VISA-RSIB object are
% identical to the binary read and write operations for the 
% VISA-GPIB, VISA-VXI, VISA-GPIB-VXI, VISA-TCPIP, and VISA-USB
% objects, except the VISA-RSIB object does not support the 
% EOSCharCode and EOSMode properties.

%% Functions and Properties
% These functions are used when reading and writing binary 
% data: 
% 
%  FREAD            - Read binary data from instrument.
%  FWRITE           - Write binary data to instrument.
% 
% These properties are associated with reading and writing
% binary data:
% 
%  ValuesReceived   - Specifies the total number of values
%                     read from the instrument.
%  ValuesSent       - Specifies the total number of values 
%                     sent to the instrument.
%  InputBufferSize  - Specifies the total number of bytes 
%                     that can be queued in the input buffer
%                     at one time.
%  OutputBufferSize - Specifies the total number of bytes 
%                     that can be queued in the output buffer
%                     at one time.
%  EOSMode          - Configures the End-Of-String termination
%                     mode.
%  EOSCharCode      - Specifies the End-Of-String terminator.

%% Creating a VISA-GPIB Object
% To begin, create a VISA-GPIB object. The board index 
% is configured to 0, the primary address is configured to 1. 
% 
%  >> g = visa('ni', 'GPIB0::1::0::INSTR');
% 
%    VISA-GPIB Object Using NI Adaptor : VISA-GPIB0-1
% 
%    Communication Address 
%       BoardIndex:         0
%       PrimaryAddress:     1
%       SecondaryAddress:   0
% 
%    Communication State 
%       Status:             closed
%       RecordStatus:       off
% 
%    Read/Write State  
%       TransferStatus:     idle
%       BytesAvailable:     0
%       ValuesReceived:     0
%       ValuesSent:         0
% 

%% Connecting the VISA-GPIB Object to the Instrument
% Before you can perform a read or write operation, you must
% connect the VISA-GPIB object to the instrument with the FOPEN
% function. If the object was successfully connected, its
% Status property is automatically configured to open. 
% 
%  >> fopen(g)
%  >> get(g, 'Status')
% 
%  ans =
% 
%  open
% 
% Note that the display summary is updated accordingly.
% 
%  >> g
% 
%    VISA-GPIB Object Using NI Adaptor : VISA-GPIB0-1
% 
%    Communication Address 
%       BoardIndex:         0
%       PrimaryAddress:     1
%       SecondaryAddress:   0
% 
%    Communication State 
%       Status:             open
%       RecordStatus:       off
% 
%    Read/Write State  
%       TransferStatus:     idle
%       BytesAvailable:     0
%       ValuesReceived:     0
%       ValuesSent:         0
% 

%% Writing Binary Data
% You use the FWRITE function to write binary data to the
% instrument. For example, the following commands will send 
% a sine wave to the oscilloscope:
% 
%  >> fprintf(g, 'Data:Destination RefB');
%  >> fprintf(g, 'Data:Encdg SRPbinary');
%  >> fprintf(g, 'Data:Width 1');
%  >> fprintf(g, 'Data:Start 1');
% 
%  >> t = (0:499) .* 8 * pi / 500;
%  >> data = round(sin(t) * 90 + 127);
%  >> cmd = double('CURVE #3500');
%  >> fwrite(g, [cmd data])
% 
% By default, the FWRITE function operates in synchronous
% mode. This means that FWRITE blocks the MATLAB command 
% line until one of the following occurs:
%
% * All the data is written
% * A timeout occurs as specified by the Timeout property
%
% By default the FWRITE function writes binary data using 
% the uchar precision. However, the following precisions 
% can also be used:
% 
%     MATLAB           Description
%     'uchar'          unsigned character,  8 bits.
%     'schar'          signed character,    8 bits.
%     'int8'           integer,             8 bits.
%     'int16'          integer,             16 bits.
%     'int32'          integer,             32 bits.
%     'uint8'          unsigned integer,    8 bits.
%     'uint16'         unsigned integer,    16 bits.
%     'uint32'         unsigned integer,    32 bits.
%     'single'         floating point,      32 bits.
%     'float32'        floating point,      32 bits.
%     'double'         floating point,      64 bits.
%     'float64'        floating point,      64 bits.
%     'char'           character,           8 bits (signed or unsigned).
%     'short'          integer,             16 bits.
%     'int'            integer,             32 bits.
%     'long'           integer,             32 or 64 bits.
%     'ushort'         unsigned integer,    16 bits.
%     'uint'           unsigned integer,    32 bits.
%     'ulong'          unsigned integer,    32 bits or 64 bits.
%     'float'          floating point,      32 bits.

%% Values Versus Bytes
% When performing a write operation, you should think of the 
% transmitted data in terms of values rather than bytes. A 
% value consists of one or more bytes. For example, one uint32  
% value consists of four bytes.

%% Binary Write Properties -- OutputBufferSize
% The OutputBufferSize property specifies the maximum number 
% of bytes that can be written to the instrument at once. 
% By default, OutputBufferSize is 512.  
% 
%  >> get(g, 'OutputBufferSize')
% 
%  ans =
% 
%      512
%  
% Configure the VISA-GPIB object's output buffer size 
% to 3000 bytes. Note, the OutputBufferSize can be configured 
% only when the object is not connected to the instrument.
% 
%  >> fclose(g); 
%  >> set(g, 'OutputBufferSize', 3000);
%  >> fopen(g); 

%% Writing Int16 Binary Data
% Now write a waveform as an int16 array.  
%
%  >> fprintf(g, 'Data:Destination RefB');
%  >> fprintf(g, 'Data:Encdg SRPbinary');
%  >> fprintf(g, 'Data:Width 2');
%  >> fprintf(g, 'Data:Start 1');
% 
%  >> t = (0:499) .* 8 * pi / 500;
%  >> data = round(sin(t) * 90 + 127);
%  >> fprintf(g, 'CURVE #3500');
% 
% Note: one int16 value consists of two bytes. Therefore, the
% following command will write 1000 bytes.
% 
%  >> fwrite(g, data, 'int16')

%% Properties Associated with Binary Writes
% The ValuesSent property indicates the total number of values
% written to the instrument since the object was connected to
% the instrument.
% 
%  >> get(g, 'ValuesSent')
% 
%  ans =
% 
%     576

%% Reading Binary Data
% You use the FREAD function to read binary data from the 
% instrument. For example, to read the waveform on channel 1
% of the oscilloscope:
% 
%  >> fprintf(g, 'Data:Source CH1');
%  >> fprintf(g, 'Data:Encdg SRPbinary');
%  >> fprintf(g, 'Data:Width 1');
%  >> fprintf(g, 'Data:Start 1');
%  >> fprintf(g, 'Curve?')
%  >> data = fread(g, 512); 
% 
% By default, the FREAD function reads data using the uchar
% precision and blocks the MATLAB command line until one of the
% following occurs:  
%
% * The specified number of values is read
% * The input buffer is filled
% * The EOI line is asserted
% * The terminator is received as specified by the EOSCharCode
%   property (if defined)
% * A timeout occurs as specified by the Timeout property 
%
% By default the FREAD function reads data using the uchar 
% precision. However, the following precisions can also be
% used:
% 
%     MATLAB           Description
%     'uchar'          unsigned character,  8 bits.
%     'schar'          signed character,    8 bits.
%     'int8'           integer,             8 bits.
%     'int16'          integer,             16 bits.
%     'int32'          integer,             32 bits.
%     'uint8'          unsigned integer,    8 bits.
%     'uint16'         unsigned integer,    16 bits.
%     'uint32'         unsigned integer,    32 bits.
%     'single'         floating point,      32 bits.
%     'float32'        floating point,      32 bits.
%     'double'         floating point,      64 bits.
%     'float64'        floating point,      64 bits.
%     'char'           character,           8 bits (signed or unsigned).
%     'short'          integer,             16 bits.
%     'int'            integer,             32 bits.
%     'long'           integer,             32 or 64 bits.
%     'ushort'         unsigned integer,    16 bits.
%     'uint'           unsigned integer,    32 bits.
%     'ulong'          unsigned integer,    32 bits or 64 bits.
%     'float'          floating point,      32 bits.

%% Values Versus Bytes
% When performing a read operation, you should think of the 
% received data in terms of values rather than bytes. A value 
% consists of one or more bytes. For example, one uint32 value
% consists of four bytes.

%% Binary Read Properties -- InputBufferSize
% The InputBufferSize property specifies the maximum number
% of bytes that you can read from the instrument. By default,
% InputBufferSize is 512.  
% 
%  >> get(g, 'InputBufferSize')
% 
%  ans =
% 
%      512
% 
% Configure the object's input buffer size to 2500.
% Note, the InputBufferSize can be configured only when the 
% object is not connected to the instrument.
% 
%  >> fclose(g); 
%  >> set(g, 'InputBufferSize', 2500);
%  >> fopen(g); 

%% Reading Int16 Binary Data
% Now read the same waveform on channel 1 as an int16  
% array.  
% 
%  >> fprintf(g, 'Data:Source CH1');
%  >> fprintf(g, 'Data:Encdg SRIbinary');
%  >> fprintf(g, 'Data:Width 2');
%  >> fprintf(g, 'Data:Start 1');
%  >> fprintf(g, 'Curve?')
% 
% Note: one int16 value consists of two bytes. Therefore, the
% following command will read 2400 bytes.
% 
%  >> data = fread(g, 1200, 'int16');

%% Binary Read Properties -- ValuesReceived
% The ValuesReceived property is updated by the number of 
% values read from the instrument.
% 
%  >> get(g, 'ValuesReceived')
% 
%  ans =
% 
%      1200

%% Binary Read Properties -- EOSMode and EOSCharCode
% For VISA-GPIB, objects, the terminator is defined by setting  
% the object's EOSMode property to read and setting the object's 
% EOSCharCode property to the ASCII code for the character 
% received. For example, if the EOSMode property is set to 
% read and the EOSCharCode property is set to 10, then one 
% of the ways that the read terminates is when the linefeed 
% character is received.
% 
% Let's configure the VISA-GPIB object's terminator to the 
% letter E.
% 
%  >> set(g, 'EOSMode', 'read');
%  >> set(g, 'EOSCharCode', double('E'));
% 
% Now, read the signal frequency of channel 1.
% 
%  >> fprintf(g, 'Measurement:Meas1:Source CH1')
%  >> fprintf(g, 'Measurement:Meas1:Type Freq')
%  >> fprintf(g, 'Measurement:Meas1:Value?')
% 
% Note, the first read terminates due to the EOSCharCode being
% detected, while the second read terminates due to the EOI 
% line being asserted.
% 
%  >> data = fread(g, 30);
%  Warning: The EOI line was asserted or the EOSCharCode was detected before SIZE values were available.
%  >> char(data)'
% 
%  ans =
% 
%  9.980040283203E
% 
%  >> data = fread(g, 30);
%  Warning: The EOI line was asserted or the EOSCharCode was detected before SIZE values were available.
%  >> char(data)' 
% 
%  ans =
% 
%  2

%% Cleanup
% If you are finished with the VISA-GPIB object, disconnect 
% it from the instrument, remove it from memory, and remove
% it from the workspace.
% 
%  >> fclose(g);
%  >> delete(g);
%  >> clear g

