% alexandre benoit : benoit.a@neuf.fr ; benoit@lis.inpg.fr ;
% www.lis.inpg.fr/pages_perso/benoit

% function that detects eye left blinks (action : opening/closing), and
% state
% using a spectrum analysis of the eye area : the OPL filter output of the
% area reports the state information, the IPLK output reports the moton
% events and its orientation
% a first eye detection is done with oriented flow pass filter computing
% (low coputation cost but can be improved)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Eye Detection and analysis

% INPUTS:
%       CurrentFrame: index of the current frame
%       FeatureType: type of the feature:1=eye, 2:mouth
%       FeatureFinderTableCoef: coefficient table for feature detection in the area (used for eye detection
%       NrowsFeature, McolumnsFeature: size in pixels of the Feature research area
%       FeatureAreaOPL: OPL filtering result of the left or right upper part of the face
%       FeatureAreaIPL: IPL filtering result--------------------------------------------
%       FeatureWindow: bump map that helps the detection of the Feature position:
%           same size as FeatureAreaOPL
%       alphaFeaturePositionMotion: value between 0 and 1 that allows the
%           temporal smoothing of the Feature position
%       FeatureUp, FeatureLeft: last position of the Feature in the Feature reseach area
%       FeatureHammingWindow: hamming window used for spectrum analysis
%       FeatureLogPolTransfoMatrixPixList, FeatureLogPolTransfoMatrixPixCoef,
%           FeatureMaxListSize: parameters of the log polar transformation
%           given by the function GloPLargeFilterInitFDivideFast2.dll
%       NBanglesFaceInside, NBradiusFaceInside: size of the output spectrum
%       picture
%       FeatureOPLTotalEnergyAdaptativeMean_Last:writne in the name
%       NoiseMeanInitDuration:duration of the initialisation part (compute
%           the noise mean)
%       FeatureNoiseMean:noise level on the feature
%
%       LastFeatureMotionEvent: frame number of the lat motion event
%       FeatureLastStableOPLValue:energy level of the OPL value of the last
%           motion free frame
%       FeatureClosedEnergyLevel: energy level that corresponds to the close feature
%       FeatureOpenedEnergyLevel: energy level that corresponds to the open feature
%       FeatureBinaryState: last state of the feature (0=close, 1=open)
%       FeatureMaxIPLEnergy: energy level used for the detection o motion alerts
%       MotionActivityExternalDetection: binary value given by an external
%           tool which allows the dtection of the sates change of the
%           feature

% OUTPUTS:
%       FeatureUp FeatureDown FeatureLeft FeatureRight:position of the bounding box of the feature in the Feature reseach area
%       FeatureState: word that shows the Feature state: 'FeatureOpen' or 'FeatureClosed'
%       FeatureMotion: value that shows the Feature action: 0='FeatureStable',
%           1='OpeningFeature' or 2='ClosingFeature'
%       FeatureOPLTotalEnergyAdaptativeMean_New:writen in the name
%       LastFeatureMotionEvent: frame number of the lat motion event
%       FeatureLastStableOPLValue:energy level of the OPL value of the last
%           motion free frame
%       FeatureClosedEnergyLevel: energy level that corresponds to the close feature
%       FeatureOpenedEnergyLevel: energy level that corresponds to the open feature
%       FeatureMaxIPLEnergy: energy level used for the detection o motion alerts
%       FeatureOPLCurrentFFT, FeatureIPLCurrentFFT: total spectrum energy
%           of the OPL and IPL outputs


function [FeatureUp FeatureDown FeatureLeft FeatureRight FeatureBinaryState FeatureMotion FeatureOPLTotalEnergyAdaptativeMean_New LastFeatureMotionEvent FeatureLastStableOPLValue FeatureClosedEnergyLevel FeatureOpenedEnergyLevel FeatureMaxIPLEnergy FeatureTotalOPLEnergy FeatureTotalIPLEnergy]=FeatureStateDetection(CurrentFrame, FeatureType, FeatureFinderTableCoef, NrowsFeature, McolumnsFeature, FeatureAreaOPL, FeatureAreaIPL, FeatureWindow, alphaFeaturePositionMotion, FeatureUp, FeatureLeft, FeatureHammingWindow, FeatureLogPolTransfoMatrixPixList, FeatureLogPolTransfoMatrixPixCoef, NBanglesFaceInside, NBradiusFaceInside, FeatureMaxListSize, FeatureOPLTotalEnergyAdaptativeMean_Last, NoiseMeanInitDuration, FeatureNoiseMean, LastFeatureMotionEvent, FeatureLastStableOPLValue, FeatureClosedEnergyLevel, FeatureOpenedEnergyLevel, FeatureBinaryState, FeatureMaxIPLEnergy, MotionActivityExternalDetection)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% variable that can be passed in argument
SignificantNumberOfFramesLevels=2;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if FeatureType==1 % case of an eye, first search the eye in the research area
    % recalculate half size of the Feature research area

    %First : filtering the Feature region
    FeatureArea = FeatureWindow.*FeatureAreaOPL;

    % Face_OPL = sortie retinaOutput
    % if CurrentFrame>20
    % figure
    % mesh(FeatureLeftArea)
    % input('press enter')
    % end
    [FeaturesSearchAreaNRows, FeaturesSearchAreaMcolumns]=size(FeatureAreaOPL);
    HalfNrowsFeature = floor(NrowsFeature/2);
    HalfMcolumnsFeature = floor(McolumnsFeature/2);

    FeatureArea_LP_H=zeros(FeaturesSearchAreaNRows, FeaturesSearchAreaMcolumns);
    FeatureArea_LP_V=zeros(FeaturesSearchAreaNRows, FeaturesSearchAreaMcolumns);

    MaxCoordFeature = SpatioTemporalSquaringLPFilterFindSymMax(abs(FeatureArea), FeatureArea_LP_H, FeatureArea_LP_V, FeaturesSearchAreaNRows, FeaturesSearchAreaMcolumns, FeatureFinderTableCoef);
    %fichier .c retourne coord du centre de l'oeil
    FeatureUp = min(FeaturesSearchAreaNRows-NrowsFeature, max(1, FeatureUp*alphaFeaturePositionMotion+(1-alphaFeaturePositionMotion)*(MaxCoordFeature(1)-HalfNrowsFeature) ));
    FeatureDown = FeatureUp+NrowsFeature-1;
    FeatureLeft = min(FeaturesSearchAreaMcolumns-McolumnsFeature, max(1, FeatureLeft*alphaFeaturePositionMotion+(1-alphaFeaturePositionMotion)*(MaxCoordFeature(2)-HalfMcolumnsFeature) ));
    FeatureRight = FeatureLeft+McolumnsFeature-1;
    %extrait le cadre de l'oeil (prend la moiti du cadre initial

    % Extract face features OPL and IPL responses
    Feature_OPL = FeatureAreaOPL(FeatureUp:FeatureDown,FeatureLeft:FeatureRight);
    Feature_IPL = FeatureAreaIPL(FeatureUp:FeatureDown,FeatureLeft:FeatureRight);
else
    % if not an eye, then, do not search, just use all the area given in input
    Feature_OPL = FeatureAreaOPL;
    Feature_IPL = FeatureAreaIPL;
    FeatureUp=FeatureUp;
    FeatureDown = FeatureUp+NrowsFeature-1;
    FeatureLeft=FeatureLeft;
    FeatureRight = FeatureLeft+McolumnsFeature-1;
end

%OPL = retina output et IPL= contours en mvt.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Blinks/Yawnings detection start  %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% FFT computing
%             a=size(FeatureHammingWindow)
%             b=size(FeatureLeft_OPL)
FeatureOPLCurrentFFT = abs(fftshift(fft2(Feature_OPL.*FeatureHammingWindow)));
FeatureIPLCurrentFFT = abs(fftshift(fft2(Feature_IPL.*FeatureHammingWindow)));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LogPolar transform using GloP filters
IPLLogPolFFTFeature=zeros(NBanglesFaceInside, NBradiusFaceInside);
FastTransfoLogPolFunctionL_C(FeatureIPLCurrentFFT, FeatureLogPolTransfoMatrixPixList, FeatureLogPolTransfoMatrixPixCoef, NBanglesFaceInside, NBradiusFaceInside, IPLLogPolFFTFeature, FeatureMaxListSize);

IPLShowLogPolSpectrum = IPLLogPolFFTFeature';

% find the max value and its coordonates
[MaxValue, IDrow]  = max(max(IPLShowLogPolSpectrum));
[MaxValue, IDcolumn]  = max(max(IPLShowLogPolSpectrum'));

% get the oriented energie at the output of the OPL and IPL filter
IPLOrientedEnergie=sum(IPLShowLogPolSpectrum);

% Computing the Total IPL energy
FeatureTotalIPLEnergy = sum(IPLOrientedEnergie);
FeatureTotalOPLEnergy = sum(sum(FeatureOPLCurrentFFT));

% estimating OPL energies
OPLAdaptativeMeanSensitivity=0.1;
FeatureOPLTotalEnergyAdaptativeMean_New = FeatureOPLTotalEnergyAdaptativeMean_Last*(1-OPLAdaptativeMeanSensitivity) + OPLAdaptativeMeanSensitivity*FeatureTotalOPLEnergy;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Modifyed by LUONG Hong Viet(June2006) & Alexandre BENOIT(July2006)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Extract motion evolution
if CurrentFrame < NoiseMeanInitDuration
    % first evaluate the noise to avoid interpretation errors
    if CurrentFrame >8
        FeatureNoiseMean = FeatureNoiseMean+FeatureTotalIPLEnergy;
    end
    % just before end of init period, fix all the values
    if CurrentFrame==(NoiseMeanInitDuration-1)
        FeatureMaxIPLEnergy = FeatureNoiseMean/(NoiseMeanInitDuration-8);
        FeatureNoiseMean = 5*FeatureNoiseMean/(NoiseMeanInitDuration-8);
        LastFeatureMotionEvent = CurrentFrame;
        FeatureClosedEnergyLevel =FeatureOPLTotalEnergyAdaptativeMean_New*0.8;
        FeatureOpenedEnergyLevel =FeatureOPLTotalEnergyAdaptativeMean_New*1.2;
        
    end
    FeatureBinaryState=0;
    FeatureMotion = 0;%'FeatureStable';

else


    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%% Evaluating variables signification
    FeatureMaxIPLEnergy = FeatureMaxIPLEnergy*0.9; % MaxEnergy decreases in the time => be aware of all events even smallest but significant...

    if (FeatureTotalIPLEnergy-FeatureNoiseMean > 0)&&(MotionActivityExternalDetection)

        % Update the max value limit :

        FeatureMaxIPLEnergy = max(FeatureMaxIPLEnergy, FeatureTotalIPLEnergy-FeatureNoiseMean);

        FeaturealphaMaxIPLEnergy = (FeatureTotalIPLEnergy-FeatureNoiseMean)/FeatureMaxIPLEnergy;
        FeaturealphaMaxIPLEnergy = FeaturealphaMaxIPLEnergy*FeaturealphaMaxIPLEnergy;
        FeaturealphaCurrentData = FeaturealphaMaxIPLEnergy;

    else
        FeaturealphaCurrentData =0;
    end


    %%%%%%%%%% Init part end %%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Detecting Feature Motion/Blink:

    % state evolution indicator pre update
    FeatureStateChange =0;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Modifier par LUONG Hong Viet
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    FeatureMotion = 0;%'FeatureStable';
    if FeaturealphaCurrentData < 0.5 || not(MotionActivityExternalDetection)
        FeatureLastStableOPLValue = FeatureOPLTotalEnergyAdaptativeMean_New;
    else
        % identifying the motion orientation
        [MaxOrientedEnergy, IndexMaxOrientedEnergy] = max(IPLOrientedEnergie);

        if abs(IndexMaxOrientedEnergy - NBanglesFaceInside/4) < NBanglesFaceInside/4

            EnergyGradient = (FeatureOPLTotalEnergyAdaptativeMean_New-FeatureOPLTotalEnergyAdaptativeMean_Last);

            if EnergyGradient < 0
                FeatureMotion = -1;%'ClosingFeature';
            else
                FeatureMotion = 1;%'OpeningFeature';
            end

            % detecting Feature state
            FeatureStableFrameNumber = CurrentFrame-LastFeatureMotionEvent;
            % init for the upcoming reinitialized records
            LastFeatureMotionEvent = CurrentFrame;

            % if there are sufficient stables frames, then compute energy levels
            if FeatureStableFrameNumber > SignificantNumberOfFramesLevels
                FeatureOpenClosedEnergyLimit = (FeatureClosedEnergyLevel+FeatureOpenedEnergyLevel)/2;
                % case of current energy level higher than the one
                % during the stable period => the Feature just closed
                if FeatureOPLTotalEnergyAdaptativeMean_New > FeatureTotalOPLEnergy
                    % updating Feature opened energy level
                    FeatureOpenedEnergyLevel = max(FeatureLastStableOPLValue, FeatureOpenClosedEnergyLimit);
                else%if OPLTotalEnergyAdaptativeMean(CurrentFrame) < TotalOPLEnergy(CurrentFrame)
                    % updating Feature opened energy level
                    FeatureClosedEnergyLevel = min(FeatureLastStableOPLValue, FeatureOpenClosedEnergyLimit);
                end

                FeatureStateChange =1;

            end
        end
    end

    % detectting Feature State
    FeatureOpenClosedEnergyLimit = (FeatureClosedEnergyLevel+FeatureOpenedEnergyLevel)/2;
    % if state change imposed, then change OK, else, continue
    % updating the last state ... try to test...goal, avoid
    % false change detection when a global head mtion occurs
    if not(MotionActivityExternalDetection)%|| FeaturesAndMouthActivity || FeaturesActivityOnly
        if FeatureTotalOPLEnergy < FeatureOpenClosedEnergyLimit
                FeatureBinaryState =0;
                FeatureClosedEnergyLevel = FeatureClosedEnergyLevel*(OPLAdaptativeMeanSensitivity) + (1-OPLAdaptativeMeanSensitivity)*min(FeatureLastStableOPLValue, FeatureOpenClosedEnergyLimit);
        else
                FeatureBinaryState =1;
                FeatureOpenedEnergyLevel = FeatureOpenedEnergyLevel*(OPLAdaptativeMeanSensitivity) + (1-OPLAdaptativeMeanSensitivity)*max(FeatureLastStableOPLValue, FeatureOpenClosedEnergyLimit);
        end
    end

    %%%  Blinks/Yawnings detection end  %%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %end of "if CurrentFrame < NoiseMeanInitDuration"
end

