% function cv = get_ad_cv(p)
%
% Get the (closest) critical value for an Anderson-Darling statistic,
% so that a test may be performed using the returned value as a
% threshold.
%
% Input parameters:
%   p -- the significance level (0 < p < 1)
%
% Outputs:
%   cv -- the critical value for the Anderson-Darling statistic at
%         significance level p
%
% $Revision: 1.2 $
% $Date: 2004/05/26 13:14:15 $
% 
% Copyright (C) 2003  Greg Hamerly (ghamerly at cs dot ucsd dot edu)
% Released under the GNU GPL software license.
% http://www.gnu.org/copyleft/gpl.html

% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of the
% License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
% 02111-1307  USA
%
% If you use this code, please let me know.
function cv = get_ad_cv(p)
    % the p-values at which we will take samples
    pvals = [0:0.001:0.09 0.10:0.01:0.94 0.95:0.001:0.998 ...
             0.999:0.0001:0.9998 0.9999:0.00001:0.99999];

    % the critical values obtained from experimentation
    ad_critical_values = [
    5.2964164776653888e-02 9.9565988295012176e-02 1.0656798880415863e-01 1.1138995321471157e-01 ...
    1.1510440104302688e-01 1.1822943902700445e-01 1.2101978729606344e-01 1.2327191161273277e-01 ...
    1.2530320508358272e-01 1.2729671165402578e-01 1.2912943401317989e-01 1.3075077319979300e-01 ...
    1.3233274413039453e-01 1.3387178342975403e-01 1.3540022748895808e-01 1.3674217115021747e-01 ...
    1.3798415505186767e-01 1.3922325824319159e-01 1.4046720321653083e-01 1.4159910436626433e-01 ...
    1.4276558860763089e-01 1.4390432777020123e-01 1.4498509963584638e-01 1.4600033501255893e-01 ...
    1.4700838635860691e-01 1.4796536243318315e-01 1.4892517253190363e-01 1.4990830848262249e-01 ...
    1.5081659300346928e-01 1.5170104824733244e-01 1.5259502429398708e-01 1.5344170478647357e-01 ...
    1.5424693954275881e-01 1.5504082994652890e-01 1.5582561962150976e-01 1.5654093860689500e-01 ...
    1.5733094474740028e-01 1.5807604484780313e-01 1.5884841026877439e-01 1.5959391609897011e-01 ...
    1.6034979809001015e-01 1.6105320363703868e-01 1.6177715738285769e-01 1.6249317536914987e-01 ...
    1.6317710424004872e-01 1.6387487503573084e-01 1.6451136737032357e-01 1.6515201546053504e-01 ...
    1.6583273933656528e-01 1.6648386554027184e-01 1.6711575569024717e-01 1.6779007308218752e-01 ...
    1.6838278066154544e-01 1.6899365315953219e-01 1.6961168971681673e-01 1.7022029355325685e-01 ...
    1.7081918163887622e-01 1.7142004930860821e-01 1.7202276623379475e-01 1.7260816294037795e-01 ...
    1.7316421962993900e-01 1.7375723651912267e-01 1.7433327250296315e-01 1.7489473130683564e-01 ...
    1.7545856591027587e-01 1.7599889332393559e-01 1.7653643077915149e-01 1.7711421879516820e-01 ...
    1.7771135050372777e-01 1.7825836031283870e-01 1.7878269512777933e-01 1.7929295286921842e-01 ...
    1.7983076926464037e-01 1.8037399385582376e-01 1.8090128209905743e-01 1.8141615320359961e-01 ...
    1.8194709553904431e-01 1.8248405263352652e-01 1.8297712296816826e-01 1.8349640622943753e-01 ...
    1.8400740051492903e-01 1.8452530295985525e-01 1.8503819842044322e-01 1.8552688730085265e-01 ...
    1.8603818584085730e-01 1.8650987520368290e-01 1.8694815953642774e-01 1.8744175823337628e-01 ...
    1.8794490547463738e-01 1.8842371913603984e-01 1.8889469481621290e-01 1.9355085411802975e-01 ...
    1.9795114023179394e-01 2.0220299589004753e-01 2.0639731676135398e-01 2.1040688884897918e-01 ...
    2.1435392107218831e-01 2.1818521284683356e-01 2.2194170237423805e-01 2.2565854841172950e-01 ...
    2.2935011985919118e-01 2.3292686886736647e-01 2.3646402714553005e-01 2.3998473007065968e-01 ...
    2.4348567034678581e-01 2.4700135445209526e-01 2.5041751066748930e-01 2.5382802576473296e-01 ...
    2.5729593794585526e-01 2.6079662951860882e-01 2.6425977070312001e-01 2.6766716507904675e-01 ...
    2.7112860029694730e-01 2.7456195360002766e-01 2.7799653171359751e-01 2.8144400772384870e-01 ...
    2.8492290606470760e-01 2.8843982541673086e-01 2.9189232250030273e-01 2.9542977216283361e-01 ...
    2.9899932133560014e-01 3.0250397597444589e-01 3.0612800829908338e-01 3.0973009345166247e-01 ...
    3.1344226249052554e-01 3.1710256459791708e-01 3.2091005516338100e-01 3.2467814565825392e-01 ...
    3.2848389662410682e-01 3.3233412740764834e-01 3.3623431411422189e-01 3.4015851482979542e-01 ...
    3.4414112847230172e-01 3.4820792515677113e-01 3.5233648088694736e-01 3.5651921828434752e-01 ...
    3.6071133250254661e-01 3.6508306383120726e-01 3.6951239917425482e-01 3.7406178446599370e-01 ...
    3.7854923723443790e-01 3.8319374431307551e-01 3.8800316071797170e-01 3.9281182857564545e-01 ...
    3.9773147140192577e-01 4.0277819334266951e-01 4.0797282599260143e-01 4.1331053910562332e-01 ...
    4.1880629254740143e-01 4.2446345527639551e-01 4.3022787542690821e-01 4.3614662385004976e-01 ...
    4.4228592274532730e-01 4.4871529727379311e-01 4.5522688606445172e-01 4.6187029699683535e-01 ...
    4.6887642748993130e-01 4.7612407936264844e-01 4.8363158874326473e-01 4.9161908333167048e-01 ...
    4.9985614862225702e-01 5.0838452923562727e-01 5.1754617622771093e-01 5.2707580142282495e-01 ...
    5.3717884353250156e-01 5.4790065253337161e-01 5.5937777282190382e-01 5.7146813297447352e-01 ...
    5.8435272489646195e-01 5.9843262944082198e-01 6.1369504291474186e-01 6.3035447859283522e-01 ...
    6.4854896206554713e-01 6.6901549802992122e-01 6.9231097086287718e-01 7.1909121389501252e-01 ...
    7.5097565768328423e-01 7.5441339187705125e-01 7.5793656021005518e-01 7.6168912475407069e-01 ...
    7.6560612697755914e-01 7.6932739327185118e-01 7.7325983572472978e-01 7.7730871622600262e-01 ...
    7.8145810028468077e-01 7.8574809240819832e-01 7.8996957752747221e-01 7.9430411191378880e-01 ...
    7.9916193189808382e-01 8.0356668219474159e-01 8.0824166692787003e-01 8.1310855702918161e-01 ...
    8.1807236121198912e-01 8.2325669638646559e-01 8.2865816161972816e-01 8.3419097000160036e-01 ...
    8.3971780678473351e-01 8.4574143011036540e-01 8.5188156800450088e-01 8.5844570170149836e-01 ...
    8.6510148449849567e-01 8.7169402130064100e-01 8.7853150743285369e-01 8.8626580711172664e-01 ...
    8.9425390144208450e-01 9.0261614014502811e-01 9.1148756860059166e-01 9.2039217573176302e-01 ...
    9.2934564974325440e-01 9.3976579681418571e-01 9.5006399683552445e-01 9.6110179251536465e-01 ...
    9.7367020446779406e-01 9.8681877500973769e-01 1.0002295775580023e+00 1.0158811643012768e+00 ...
    1.0329558308872038e+00 1.0513436717989180e+00 1.0725767207350145e+00 1.0959266358321429e+00 ...
    1.1231067835547037e+00 1.1564707409145285e+00 1.1971354990944292e+00 1.2482137171818977e+00 ...
    1.3170163298732405e+00 1.4398289201598118e+00 1.4565556944346554e+00 1.4760482684316685e+00 ...
    1.5015178727561533e+00 1.5289623796511478e+00 1.5654766342884159e+00 1.5995584071301892e+00 ...
    1.6572611751222439e+00 1.7258182306944718e+00 1.8525117794506514e+00 1.8732492174957542e+00 ...
    1.8913738918420222e+00 1.9281457625330631e+00 1.9506717238492683e+00 1.9831784543312097e+00 ...
    2.0238553889753348e+00 2.0727500521954880e+00 2.1654903296303019e+00 2.3572861687715658e+00
    ];

    cv = zeros(size(p));
    for i = 1:length(p)
        ndx = max(find(pvals < p(i)));
        if (isempty(ndx))
            ndx = 1;
            p(i) = pvals(1);
        elseif (ndx == length(pvals))
            ndx = length(pvals) - 1;
            p(i) = pvals(end);
        end

        % perform a linear interpolation between the two surrounding values
        weight = 1 - abs(pvals(ndx) - p(i)) / range(pvals(ndx:ndx+1));
        cv(i) = ad_critical_values(ndx) * weight + ad_critical_values(ndx+1) * (1 - weight);
    end;

