/* Jannis Agiomyrgiannakis and Andre Holzapfel */
/* Signal Processing Laboratory */
/* University of Crete, June 2006 */
/* Supervisor: Jannis Stylianou */
/* */
/* Enterface06, Multimodal Character Morphing Project */ 

#include "cvq_lib.h"
/*#include <stdio.h>*/
/*#include <math.h>*/

char error_txt[512] = "undefined";


void cvq_train_hard(double *mY, DTYPE *X, DTYPE *Y, int P, int D, int N, int K, int M, double *mX, double dthres, int maxiter, int settings)
{
	int n, m, p, d, k;
	int *L, Ns;
	double **mY_ptr;
	DTYPE  *Ys, *Ys_ptr, *Y_ptr;

	if(settings & CVQ_SETTINGS_VERBOSE) printf("cvq_train_hard: P = %d, D = %d, N = %d, M = %d, K = %d, dthres = %.3e, maxiter = %d\n",P,D,N,M,K,(float)dthres,maxiter);
	L = (int *)calloce(N,sizeof(int));
	Ys = (DTYPE *)calloce(D*N,sizeof(DTYPE));
	if(settings & CVQ_SETTINGS_VERBOSE) printf("cvq_train_hard: computing codebook labels\n");
	vq_encode_eucl(L, X, mX, P, M, N, NULL, 0);
	mY_ptr = &mY;
	for(m=0;m<M;m++)
	{
		/* isolate the Y|X-space vectors of the specific m-th X-space class */
		Ys_ptr = Ys;
		Y_ptr = Y;
		Ns = 0;
		for(n=0;n<N;n++)
		{
			if(L[n]==m)
			{
				for(d=0;d<D;d++)	Ys_ptr[d] = Y_ptr[d];
				Ys_ptr += D; Ns++;
			}
			Y_ptr += D;
		}
		/* slbg vq for the specific Y|X-space class */
		if(settings & CVQ_SETTINGS_VERBOSE)
		{
			printf("(class %d: %d vectors)\t",m,Ns);
			if(m%4==3) printf("\n");
			fflush(stdout);
		}
		slbg(Ys, mY_ptr, &K, 1, D, Ns, dthres, maxiter, SLBG_SETTINGS_SIMPLE_SPLIT, NULL);
		*mY_ptr += D*K;
	}
	/* printf("\n"); */

	free(L);
	free(Ys);
}

void slbg(DTYPE *X, double **CBK, int *MM, int nM, int P, int N, double dthres, int maxiter, int settings, double *W)
{
	slbg_fp(X, CBK, MM, nM, P, N, dthres, maxiter, settings, W, NULL, 0);
}

void vq_encode_eucl(int *L, DTYPE *X, double *CBK, int P, int M, int N, double *D, int sqr_flag)
{
    int n, m, p, min_cell;
    double dist, mindist, tmp, *CBKn;
    DTYPE *Xn;

    Xn = X;
    for(n=0;n<N;n++) {
        CBKn = CBK;
        mindist = 1e10;
        min_cell = 0;
        for(m=0;m<M;m++)
		{
            dist = 0.0;
            for(p=0;p<P;p++)
            {
                tmp = Xn[p] - CBKn[p];
                dist += tmp*tmp;
            }
	        if(sqr_flag==0) dist = sqrt(dist);
            if(dist < mindist) { mindist = dist; min_cell = m; }
            CBKn += P;
        }
        L[n] = min_cell;
		if(D!=NULL) D[n] = mindist;
        Xn += P;
    }
}

void slbg_fp(DTYPE *X, double **CBK, int *MM, int nM, int P, int N, double dthres, int maxiter, int settings, double *W, double *Cfp, int Mfp)
{
	int    i, iteration, MMi, *L, maxM, M, numel;
	double *C, dist, prev_dist, dim_dist, epsilon;
	char   tmpfname[1024];

	sprintf(error_txt,"slbg: P = %d, N = %d",P,N);
	for(i=0;i<nM;i++)
		if(MM[i]>16*1024 || MM[i]<=0 || power_of_2(MM[i])==0)
			error3("requested codebook size is out of bounds, or it is not a power of two",MM[i]);
	for(i=1;i<nM;i++)
		if(MM[i-1]>=MM[i]) error4("requested codebook sizes are not sorted",MM[i-1],MM[i]);
	MMi = 0;
	maxM = MM[nM-1];
	C = (double *)calloce(maxM*P,sizeof(double));
	L = (int *)calloce(N,sizeof(int));
	/* initialization */
	for(i=0; i<N; i++) L[i] = 0;
	if(Cfp==NULL)
	{
		/* initialize single entry codebook */
		M = 1;
		slbg_compute_centroids(C, P, M, N, X, L, W);
		dist = slbg_average_distortion(X, L, C, P, M, N, W);
		if(MM[MMi] == M)
		{
			/* store result to memory */
			if(CBK[MMi]==NULL)	{ CBK[MMi] = (double *)calloce(M*P,sizeof(double)); printf("load"); }
			copy(CBK[MMi], C, M*P);
			MMi++;
		}
	} else {
		/* the starting codebook is the fixed codebook with a slight perturbation */
		M = (int)pow(2,floor(log2((double)Mfp))+1);
		if(M>=maxM) M = maxM;
		numel = P*Mfp;
		if(settings & SLBG_SETTINGS_VERBOSE) printf("fixed codebook: M = %d, maxM = %d\n",Mfp,maxM);
		init_genrand(219863248);
		for(i=0;i<P*M;i++) {
			epsilon = (((double)genrand_int32())/((double)0xffffffff))-0.5;
			C[i] = Cfp[i%numel] + 0.000001*epsilon;
			/*C[i] = epsilon;*/
		}	
		/*display(C,P,M);	display(Cfp,P,Mfp);*/
	}
	while(M<maxM)
	{
		if(settings & SLBG_SETTINGS_SIMPLE_SPLIT) 	M = slbg_split(C, P, M); else
		if(settings & SLBG_SETTINGS_EIGEN_SPLIT) 	M = slbg_split_eig(X, L, C, P, M, N, W); else
		error("incorrect settings");
		if(settings & SLBG_SETTINGS_VERBOSE) printf("Processing> %d vectors...\n",M);
		dist = kmeans_fp(X, C, P, M, N, L, dthres, maxiter, settings & SLBG_SETTINGS_VERBOSE, W, Cfp, Mfp);
		if(MM[MMi] == M)
		{
			/* store result to memory */
			if(CBK[MMi]==NULL)	{ CBK[MMi] = (double *)calloce(M*P,sizeof(double)); }
			copy(CBK[MMi], C, M*P);
			MMi++;
		}
	}
	if(settings & SLBG_SETTINGS_VERBOSE) printf("----------- final k-means iterations: (%d vectors) -------------\n",M);
	dist = kmeans_fp(X, C, P, M, N, L, dthres, maxiter, settings & SLBG_SETTINGS_VERBOSE, W, Cfp, Mfp);
	if(MM[MMi] == M)
	{
		/* store result to memory */
		if(CBK[MMi]==NULL)	{ CBK[MMi] = (double *)calloce(M*P,sizeof(double)); }
		copy(CBK[MMi], C, M*P);
	}
	free(C);
	free(L);
}

void * calloce(int N, int Nsize)
{
	void *ptr;	
	if((ptr=calloc(N,Nsize))==NULL) error3("could not allocate memory",N*Nsize);
	return ptr;
}

int slbg_split(double *C, int P, int M)
{
	int m, p;
	double *Mptr, *M2ptr, value;

	Mptr = C + (M-1)*P;
	M2ptr = C + (2*M - 1)*P;
	for(m=0;m<M;m++)
	{
		for(p=0;p<P;p++)
		{
			value = Mptr[p];
			M2ptr[p] = value + SLBG_SPLIT_PERTURBATION;
			M2ptr[p-P] = value - SLBG_SPLIT_PERTURBATION;
		}
		Mptr -= P;
		M2ptr -= P+P;
	}
	M = 2*M;
	return M;
}

int slbg_split_eig(DTYPE *X, int *L, double *C, int P, int M, int N, double *W)
{
        int i, j, m, n, p, cc, index, nrot;
        double **DataMatrix, *EigenVectors, *EigenValues;
        double *DM, *tmpX, *mX, *PD, *eV; /* [PxM] best principal direction for each class */
        double *Mptr, *M2ptr;
        double value, Wn, tmp;
		DTYPE  *cX;

        /*
        //      allocate & initialize memory
        */
        DataMatrix = (double **)calloce(M,sizeof(double *));
		PD = (double *)calloce(P*M,sizeof(double));
		tmpX = (double *)calloce(P,sizeof(double));
        EigenValues = (double *)calloce(P*P,sizeof(double));
        EigenVectors = (double *)calloce(P*P,sizeof(double));
        for (m=0; m<M; m++)
        {
                DataMatrix[m] = (double *)calloce(P*P,sizeof(double));
                for (i=0;i<P*P;i++)	DataMatrix[m][i] = 0;
        }

        /*
        //      compute best principal direction for each class
        */
        /* compute the data matrix */
        for (n=0; n<N; n++)
        {
                /* taking the class indexes */
				if(W==NULL) Wn = 1.0; else Wn = W[n];
                cc = L[n];
				if(L[n]>=0)
				{
					DM = DataMatrix[cc];
					cX = X + n*P;   /* current X vector */
					mX = C + cc*P;  /* current class mean vector */
					for (p=0; p<P; p++)     tmpX[p] = cX[p] - mX[p];
					for (i=0; i<P; i++)
						for (j=0; j<P; j++)
							DM[i*P+j] += Wn*tmpX[i]*tmpX[j];
				}
        }

        /* find the eigen vectors for the data matrix of each class */
        for (m=0; m<M; m++)
        {
                DM = DataMatrix[m];
				eigenvector_sym(DM, P, EigenVectors, EigenValues);
				/*display(EigenValues,1,P);*/
                for(p=0;p<P;p++)	PD[m*P+p] = EigenVectors[(P-1)*P+p];
        }

        /*
        // now we can start spliting
        */
		Mptr = C + (M-1)*P;
        M2ptr = C + (2*M-1)*P;
		eV = PD + (M-1)*P;
        for(m=0;m<M;m++)
        {
                for(p=0;p<P;p++)
                {
                        value = Mptr[p];
                        tmp = eV[p]*SLBG_SPLIT_PERTURBATION;
                        M2ptr[p] = value + tmp;
                        M2ptr[p-P] = value - tmp;
                }
                eV -= P;
                Mptr -= P;
                M2ptr -= P+P;
        }

        /*
        // freeing memory
        */
        free(EigenValues);
        free(EigenVectors);
        for (m=0; m<M; m++)		free(DataMatrix[m]);
        free(DataMatrix);
		free(PD);
		free(tmpX);

        /* now double the num of classes */
        M = 2*M;
		return M;
}

void eigenvector_sym(double *A, int P, double *V, double *D)
{
	char jobz, uplo, txt[128];
	int i, lwork, info;
	double *work;

	
	lwork = (P+2)*P;
	work = (double *)calloc(lwork+1,sizeof(double));
	jobz = 'V';
	uplo = 'U';
	for(i=0;i<P*P;i++) V[i] = A[i];
	dsyev_ (&jobz, &uplo, &P, V, &P, D, work, &lwork, &info);
	if(info!=0) error("eigenvector_sym: algorithm failed to converge\n");
	free(work);
}

int power_of_2(int x)
{
	if(x<0) x = -x;
	while(x>1)
	{
		if(x%2!=0) return 0;
		x = x/2;
	}

	return 1;
}


double kmeans_fp(DTYPE *X, double *C, int P, int M, int N, int *L, double dthres, int maxiter, int verbose, double *W, double *Cfp, int Mfp)
{
	double dist, prev_dist, dim_dist;
	int iteration, free_L=0;

	if(verbose) printf("kmeans (with fixed points): P = %d, M = %d, N = %d, dthres = %.6f, maxiter = %d\n",P,M,N,(float)dthres,maxiter);
	if(L==NULL) { L = (int *)calloce(N,sizeof(int)); free_L=1; }
	if(Cfp!=NULL)
		slbg_labeling_fp(L, X, C, P, M, N, Cfp, Mfp);
	else
		slbg_labeling(L, X, C, P, M, N);		
	prev_dist = 10e200;
	dist = slbg_average_distortion(X, L, C, P, M, N, W);
	dim_dist = 1;
	iteration=0;
	while(dim_dist > dthres)
	{
		if (iteration >= maxiter) break;
		slbg_compute_centroids(C, P, M, N, X, L, W);
		if(Cfp!=NULL)
			slbg_labeling_fp(L, X, C, P, M, N, Cfp, Mfp);
		else
			slbg_labeling(L, X, C, P, M, N);
		prev_dist = dist;
		dist = slbg_average_distortion(X, L, C, P, M, N, W);
		dim_dist = fabs((prev_dist-dist)/prev_dist);
		iteration++;
		if (verbose)  printf("\t> Av. Distorsion: %G (- %.3lf%%)\n", dist, dim_dist*100);
	}
	if(free_L==1) free(L);
	return dist;
}

double slbg_average_distortion(DTYPE *X, int *L, double *C, int P, int M, int N, double *W)
{
	int i, p;
	double Wn, tmp, sumo, dist=0.0;
	double *Cp, sumW;
	DTYPE *Xp;

	Xp = X;
	sumW = 0.0;
	for(i=0;i<N;i++)
	{
		if(W!=NULL) Wn = W[i]; else Wn = 1.0;
		if(L[i]>=0)
		{
			Cp = C+L[i]*P;
			sumo = 0.0;
			for(p=0;p<P;p++)
			{
				tmp = Cp[p] - Xp[p];
				sumo += tmp*tmp;
			}
			dist += Wn*sumo;
			sumW += Wn;
		}
		Xp += P;
	}
	dist /= sumW;
	return dist;
}

void slbg_labeling(int *L, DTYPE *X, double *C, int P, int M, int N)
{
    int n, m, p, min_cell;
    double dist, mindist, tmp, *Cm;
    DTYPE *Xn;

    Xn = X;
    for(n=0;n<N;n++) {
        Cm = C;
        mindist = 1e10;
        min_cell = 0;
        for(m=0;m<M;m++)
		{
            dist = 0.0;
            for(p=0;p<P;p++)
            {
                tmp = Xn[p] - Cm[p];
                dist += tmp*tmp;
            }
           if(dist < mindist) { mindist = dist; min_cell = m; }
            Cm += P;
        }
        L[n] = min_cell;
        Xn += P;
    }
}


void slbg_labeling_fp(int *L, DTYPE *X, double *C, int P, int M, int N, double *Cfp, int Mfp)
{
    int n, m, p, min_cell;
    double dist, mindist, tmp, *Cm;
    DTYPE *Xn;

    Xn = X;
    for(n=0;n<N;n++) {
        Cm = C;
        mindist = 1e10;
        min_cell = 0;
        for(m=0;m<M;m++)
		{
            dist = 0.0;
            for(p=0;p<P;p++)
            {
                tmp = Xn[p] - Cm[p];
                dist += tmp*tmp;
            }
	        /* dist = sqrt(dist); */
            if(dist < mindist) { mindist = dist; min_cell = m; }
            Cm += P;
        }
		Cm = Cfp;
        for(m=0;m<Mfp;m++)
		{
            dist = 0.0;
            for(p=0;p<P;p++)
            {
                tmp = Xn[p] - Cm[p];
                dist += tmp*tmp;
            }
	        /* dist = sqrt(dist); */
            if(dist < mindist) { mindist = dist; min_cell = -1; }
            Cm += P;
        }		
        L[n] = min_cell;
        Xn += P;
    }
}

void slbg_compute_centroids(double *C, int P, int M, int N, DTYPE *X, int *L, double *W)
{
	int i,j, imax, offset;
	double Wn, *buf_vec, *buf, *counter, *counter_copy, tmp;
	DTYPE *data_vec;

	counter = (double *)calloce(M+M,sizeof(double));
	counter_copy = counter + M;
	buf = (double *)calloce(M*P,sizeof(double));
	for(i=0;i<M;i++)      counter[i] = 0.0;
	for(i=0;i<M*P;i++)    buf[i] = 0.0;
	for(i=0;i<N;i++)
	{
		if(W==NULL) Wn = 1.0; else Wn = W[i];
		if(L[i]>=0) {
			buf_vec = buf + L[i]*P;
			data_vec = X + i*P;
			for(j=0;j<P;j++)
			{
				buf_vec[j] += Wn*data_vec[j];
			}
			counter[L[i]] += Wn;
		}
	}
	/*display(counter,1,M);*/
	for(i=0;i<M;i++)      counter_copy[i] = counter[i];
	for(i=0;i<M;i++)
	{
		tmp = counter[i];
		if (tmp==0)
		{
			printf("Splitting caused an outlier (M = %d,  Empty class index = %d)\n", M, i);
			/* find the biggest class */
			imax = find_max_ind(counter_copy, M);
			printf("Correction: split the biggest class (biggest class index = %d)",imax);
			/* add perturbation */
			for(j=0;j<P;j++)	C[i*P+j] = (buf[imax*P+j]/counter[imax]) + 10e-10;			
			counter_copy[imax] = 0; /* make sure that this class will not be splitted in the near future */
			/*
			error("slbg_compute_centroids: splitting caused an outlier\n");
			exit(1);
			*/
		} else {
			for(j=0;j<P;j++)
			{
				C[i*P+j] = buf[i*P+j]/tmp;
			}
		}
	}
	free(counter);
	free(buf);
}

int find_max_ind(double *X, int N)
{
	int n, max_ind;
	double maxval;

	maxval = -10e40;	
	for(n=0;n<N;n++) if(maxval<X[n]) { maxval=X[n]; max_ind = n; };
	
	return max_ind;
}


/* THIS IS THE RANDOM GENERATOR */
#define N 624
#define M 397
#define MATRIX_A 0x9908b0dfUL   
#define UPPER_MASK 0x80000000UL 
#define LOWER_MASK 0x7fffffffUL 
static unsigned long mt[N]; /* the array for the state vector  */
static int mti=N+1; /* mti==N+1 means mt[N] is not initialized */

/* initializes mt[N] with a seed */
void init_genrand(unsigned long s)
{
    mt[0]= s & 0xffffffffUL;
    for (mti=1; mti<N; mti++) {
        mt[mti] = 
	    (1812433253UL * (mt[mti-1] ^ (mt[mti-1] >> 30)) + mti); 
        /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
        /* In the previous versions, MSBs of the seed affect   */
        /* only MSBs of the array mt[].                        */
        /* 2002/01/09 modified by Makoto Matsumoto             */
        mt[mti] &= 0xffffffffUL;
        /* for >32 bit machines */
    }
}

/* generates a random number on [0,0xffffffff]-interval */
unsigned long genrand_int32(void)
{
    unsigned long y;
    static unsigned long mag01[2]={0x0UL, MATRIX_A};
    /* mag01[x] = x * MATRIX_A  for x=0,1 */

    if (mti >= N) { /* generate N words at one time */
        int kk;

        if (mti == N+1)   /* if init_genrand() has not been called, */
            init_genrand(5489UL); /* a default initial seed is used */

        for (kk=0;kk<N-M;kk++) {
            y = (mt[kk]&UPPER_MASK)|(mt[kk+1]&LOWER_MASK);
            mt[kk] = mt[kk+M] ^ (y >> 1) ^ mag01[y & 0x1UL];
        }
        for (;kk<N-1;kk++) {
            y = (mt[kk]&UPPER_MASK)|(mt[kk+1]&LOWER_MASK);
            mt[kk] = mt[kk+(M-N)] ^ (y >> 1) ^ mag01[y & 0x1UL];
        }
        y = (mt[N-1]&UPPER_MASK)|(mt[0]&LOWER_MASK);
        mt[N-1] = mt[M-1] ^ (y >> 1) ^ mag01[y & 0x1UL];

        mti = 0;
    }
  
    y = mt[mti++];

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return y;
}
