/* Jannis Agiomyrgiannakis and Andre Holzapfel */
/* Signal Processing Laboratory */
/* University of Crete, June 2006 */
/* Supervisor: Jannis Stylianou */
/* */
/* Enterface06, Multimodal Character Morphing Project */ 

#ifndef CVQ_LIB_H
#define CVQ_LIB_H

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>

#include "mex.h"
#include "matrix.h"
/*#include "math.h"*/

	#define error(txt) mexErrMsgTxt(txt)
	#define error2(txt) { sprintf(error_txt,"%s: ERROR: %s",error_txt,txt); mexErrMsgTxt(error_txt); }
	#define error3(txt,P) { sprintf(error_txt,"%s: ERROR: %s (%d)",error_txt,txt,P); mexErrMsgTxt(error_txt); }
	#define error4(txt,P1,P2) { sprintf(error_txt,"%s: ERROR: %s (%d of %d)",error_txt,txt,P1,P2); mexErrMsgTxt(error_txt); }
	#define error_s(txt,s1) { fprintf(stderr,"%s: ERROR: %s (`%s`)\n",error_txt,txt,s1); mexErrMsgTxt(error_txt); }
	#define error_ss(txt,s1,s2) { fprintf(stderr,"%s: ERROR: %s (`%s`) (`%s`)\n",error_txt,txt,s1,s2); mexErrMsgTxt(error_txt); }
	#define error_bound(txt,i1,i2,i3) { fprintf(stderr,"%s: ERROR: %s (%d) out of bounds [%d %d ]\n",error_txt,txt,i2,i1,i3); mexErrMsgTxt(error_txt); }
	#define DTYPE	double

#define PI  3.141592653589793115997963468544185161590576171875
#define EXP 2.71828182845904553488480814849026501178741455078125
#define FALSE	0
#define	TRUE	1
#define IS_NAN(x) (x!=x)
#define SQRT_2	1.4142135623730951454746218587388284504413604736328125
#define iSQRT_2	0.707106781186547461715008466853760182857513427734375



#ifdef SOFT_LOG2
	#define INV_LOG_2	1.4426950408889634
	#define log2(x) (log(x)*INV_LOG_2)
#endif

/* TABLE DECLARATIONS */
extern double normcdf_table[];
#define normcdf_table_N  		4096
#define normcdf_table_xstart 	-4
#define normcdf_table_xend 		4
#define normcdf_lint(y,x) linterp1(y,x,normcdf_table_xstart,normcdf_table_xend,normcdf_table,normcdf_table_N)

#define BASIC_DECLARATIONS(DTYPE)	int _i, _j, _p, _d, _m, _k, _l; DTYPE *_A, *_B, *_C, _tmp, _tmp1, _tmp2, _tmp3, _tmp4;


#define	CVQ_SETTINGS_SIMPLE_SPLIT	SLBG_SETTINGS_SIMPLE_SPLIT
#define	CVQ_SETTINGS_EIGEN_SPLIT	SLBG_SETTINGS_EIGEN_SPLIT
#define CVQ_SETTINGS_VERBOSE		SLBG_SETTINGS_VERBOSE
#define SLBG_SPLIT_PERTURBATION	0.000000001

#define SLBG_DEFAULT_MAXITER		40
#define SLBG_DEFAULT_DTHRES			0.0000001
#define	SLBG_SETTINGS_SIMPLE_SPLIT	0x10000000
#define	SLBG_SETTINGS_EIGEN_SPLIT	0x20000000
#define SLBG_SETTINGS_VERBOSE		0x40000000

#define SLBG_SETTINGS_DEFAULT		(SLBG_SETTINGS_VERBOSE | SLBG_SETTINGS_SIMPLE_SPLIT)

#define CVQ_SETTINGS_DEFAULT		(SLBG_SETTINGS_DEFAULT)
#define cbk_labeling	vq_encode_eucl

#define GAUSSIAN_DISCARD_THRES		0.05
#define CBK_FILE_HEADER_MAGICTXT	"CBK_FILE"

#define copy(target,source,N) for(i=0;i<N;i++) target[i] = source[i];




/** Conditional VQ training, based on VQ hard partitioning of the input space.
 *	@param X		[PxN] (input) N P-dimensional X-space samples
 *	@param Y		[DxN] (input) N D-dimensional Y-space samples
 *	@param K		the requested number of Y|X-space samples
 *	@param mX		[PxM] (input) it contains the GMM centers, or the VQ codebook
 *	@param dthres	the relative distortion reduction for SLBG convergence
 *	@param maxiter	the maximum number of iterations for SLBG convergence
 *	@param settings settings that control the usage of the class
 *	@param mY		[DxM*K] (output) the Y|X-space codebooks
 */
void cvq_train_hard(double *mY, DTYPE *X, DTYPE *Y, int P, int D, int N, int K, int M, double *mX, double dthres, int maxiter, int settings);

/** split LBG algorithm
 *  *	@param	X:		[PxN]	(input) the N P-dimensional samples
 *   *	@param	CBK:	[1xnM] arrays of size [PxMM(0)], [PxMM(1)], ..., [PxMM(nM-1)] (input/output).
 *    *					Each array holds the resulting codebook of size M. If CBK[i]==NULL
 *     *					the subroutine allocates memory via malloc(.).
 *      *	@param	MM:		[1xnM] 	(input) the resulting codebook sizes
 *       *	@param	dthres:		distortion ratio threshold for convergence
 *        *	@param	maxiter:	maximum number of iterations for convergence
 *         */
void slbg(DTYPE *X, double **CBK, int *MM, int nM, int P, int N, double dthres, int maxiter, int settings, double *W);

/** finds the nearest codevector for each datavector
 *  *  @param L			[1xN] (output) the indices of the nearest class {0...M-1}
 *   *	@param X			[PxN] (input) the data vectors
 *    *	@param CBK			[PxM] (input) the codebook
 *     *	@param D			[1xN] (optional output) the distance from the nearest vector. disable: set=NULL
 *      *  @param sqr_flag		if ==1 then the square of the euclidean distance is returned
 *       */
void vq_encode_eucl(int *L, DTYPE *X, double *CBK, int P, int M, int N, double *D, int sqr_flag);

/** split LBG algorithm, which also receives a fixed codebook
 *  *	@param	X:		[PxN]	(input) the N P-dimensional samples
 *   *	@param	CBK:	[1xnM] arrays of size [PxMM(0)], [PxMM(1)], ..., [PxMM(nM-1)] (input/output).
 *    *					Each array holds the resulting codebook of size M. If CBK[i]==NULL
 *     *					the subroutine allocates memory via malloc(.).
 *      *	@param	MM:		[1xnM] 	(input) the resulting codebook sizes
 *       *	@param	dthres:		distortion ratio threshold for convergence
 *        *	@param	maxiter:	maximum number of iterations for convergence
 *         */
void slbg_fp(DTYPE *X, double **CBK, int *MM, int nM, int P, int N, double dthres, int maxiter, int settings, double *W, double *Cfp, int Mfp);

/** error checking calloc routine
 *  */
void * calloce(int N, int Nsize);

extern char error_txt[];

/** produces a double sized codebook, by splitting each codevector with a small perturbation
 *  *	@param C:	[PxM] (input/output) the codebook
 *   *	@return the new codebook size (2*M)
 *    *//** splits a codebook in two parts
 *	@param C:	[PxM] (input/output) the codebook
 *	 *	@return the new codebook size (2*M)
	*	  */
int slbg_split(double *C, int P, int M);
	
/** produces a double sized codebook, by splitting each codevector
 *  *  with a small perturbation at the principal direction
 *   *	@param X:	[PxN] (input) the N P-dimensional data vectors
 *    *	@param L:	[1xN] (input) the indices describing class memberships
 *     *	@param C:	[PxM] (input/output) the codebook
 *      *	@param W:	[1xN] the weight of each sample
 *       *	@return the new codebook size (2*M)
 *        */
int slbg_split_eig(DTYPE *X, int *L, double *C, int P, int M, int N, double *W);

/** compute eigenvectors and eigenvalues for a symmetric matrix A
 *  *	@param A 	[PxP] (input) symmetric matrix
 *   *	@param V	[PxP] (output) eigenvectors (columnwise)
 *    *	@param D	[Px1] (output) corresponding eigenvalues (in ascending order)
 *     */
void eigenvector_sym(double *A, int P, double *V, double *D);

/** returns 1 if <x> is a power of 2, 0, otherwise
 *  */
int power_of_2(int x);

/** generates a random number on [0,0xffffffff]-interval */
unsigned long genrand_int32(void);

/** initializes Mersenne twister with a seed */
void init_genrand(unsigned long s);
#define init_rnd_seed	init_genrand

/** simple kmeans algorithm, which also receives a fixed codebook 
 *  *	@param	X:		[PxN]	(input) the N P-dimensional samples
 *   *	@param	C:		[PxM]	(output) the codebook
 *    *	@param	Cfp:	[PxMfp]	(input) the fixed codebook 
 *     *	@param	L:		[1xN] 	memory allocated externally for the indices
 *      *                          (optional: if==NULL, memory is allocated internally)
 *       *	@param	dthres:		distortion ratio threshold for convergence
 *        *	@param	maxiter:	maximum number of iterations for convergence
 *          *	@param 	W:	[1xN] the weight of each sample
 *           *	@return	the distortion achieved
 *            */
double kmeans_fp(DTYPE *X, double *C, int P, int M, int N, int *L, double dthres, int maxiter, int verbose, double *W, double *Cfp, int Mfp);

/** computes the average distortion of a certain codebook
 *  *	@param X:	[PxN] the P-dimensional N samples
 *   *	@param L:	[1xN] the indices of each class
 *    *	@param C:	[PxM] the codebook
 *     *	@param W:	[1xN] the weight of each sample
 *      *	@return the average distortion (square euclidean distance)
 *       */
double slbg_average_distortion(DTYPE *X, int *L, double *C, int P, int M, int N, double *W);

/** finds the nearest codevector for each datavector, but also receives a fixed codebook.
 *  *	if a point X(n) is nearest to a fixed codeword, then L(n) is set to -1
 *   *  @param L	[1xN] (output) the indices of the nearest class
 *    *	@param X	[PxN] (input) the data vectors
 *     *	@param C	[PxM] (input) the codebook
 *      *	@param Cfp:	[PxMfp]	(input) the fixed codebook 
 *       */
void slbg_labeling_fp(int *L, DTYPE *X, double *C, int P, int M, int N, double *Cfp, int Mfp);

/** finds the nearest codevector for each datavector
 *  *  @param L	[1xN] the indices of the nearest class
 *   *	@param X	[PxN] the data vectors
 *    *	@param C	[PxM] the codebook
 *     */
void slbg_labeling(int *L, DTYPE *X, double *C, int P, int M, int N);

/** computes the centroids of each class, according to the labels
 *  *	@param X:	[PxN] (input) the P-dimensional N samples
 *   *	@param L:	[1xN] (input) the indices of each class
 *    *	@param C:	[PxM] (output) the codebook
 *     *	@param W:	[1xN] the weight of each sample
 *      */
void slbg_compute_centroids (double *C, int P, int M, int N, DTYPE *X, int *L, double *W);

/** finds the index of the maximum value
 *  *	@param X	[1xN] data
 *   *	@return		the index of the maximum value
 *    */
int find_max_ind(double *X, int N);


#endif
