function centers = NM4M_KMeans(data, endK,alpha, splitMethod, testMethod, callback)

%kmeans algorithm using splitting from one cluster up to k. Code derived
%from the gmeans algorithm implemented by Greg Hamerly.
%Inputs:
%data           data matrix with data in its ROWS!
%endK           stop splitting when this given number of clusters is
%               reached
%alpha          parameter to determine the critical value for the
%               significance test
%splitMethod    use either pca or random splitting
%testMethod     Anderson-Darling:'ad', otherwise Gamma distribution test
%callback       function to be called from out of this function
%Outputs:
%centers        the cluster centers
%
%
% Andre Holzapfel
% Master Thesis Implementation
% Signal Processing Laboratory
% University of Crete, June 2006
% Supervisor: Jannis Stylianou

    if (nargin < 6) callback = 0;        end;
    if (nargin < 5) testMethod = 'ad';   end; % default: Anderson-Darling
    if (nargin < 4) splitMethod = 'pca'; end; % default: use PCA splitting
    if (nargin < 3) alpha = 0.001;   end; % default: alpha = 0.1%

    if (strcmp(testMethod, 'ad')) % find the anderson-darling critical value
        adcv = get_ad_cv(1 - alpha);
    end;

    % be quiet about Ritz values when doing eigenvalue decomposition
    opts.disp = 0; 

    [n, d] = size(data);
    ctrs = mean(data); % start with one center at the mean
    oldK = size(ctrs, 1);
    newK = 0;

    centerId = 0;
    rejectedSplits = []; % centers we should not split again
    centerIds = [centerId];

    iter = 0;

    while (oldK ~= newK)
        iter = iter + 1;
        
        if (callback ~= 0)
            eval([callback, '(ctrs, data, iter);']);
        end;

        oldK = size(ctrs, 1);
        newCtrs = [];
        newCtrIds = [];
        ind = [];
        AndersonStat = zeros(1,oldK);
        labels = findlabels(ctrs, data);
        fc = zeros(2,d,oldK);

        for i = 1:oldK
            m(i) = sum(labels == i);
        end;

        for i = 1:oldK
            c = ctrs(i,:);

            % do not re-consider splits which have already
            % been rejected
            if (sum(rejectedSplits == centerIds(i)) > 0)
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                continue;
            end;
        
            if (m(i) == 0) % delete centers with zero member points
                continue;
            elseif (m(i) == 1) % special case for centers with one member point
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                continue;
            end;

            cdata = data(labels == i, :);
            cdist = distortion(cdata);

            % if the distortion is too small, then don't try to split
            if (cdist < eps)
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                continue;
            end;

            covariance = cov(cdata);

            % RANDOM initialization
            if (strcmp(splitMethod, 'random'))
                offset = (rand(1,d)-0.5)*sqrt(trace(covariance))*0.1;
            % PCA initialization
            else
                % get the principal component
                [eigvecs, eigvals] = eig(covariance);
                [eigval, maxNdx] = max(diag(eigvals));
                eigvec = eigvecs(:,maxNdx);
                offset = eigvec' * sqrt(2 * eigval / pi);
            end;

            % create the two child centers
            ci = [c - offset; c + offset];

            % run k-means for 2 centers
            fc(:,:,i) = km(cdata, ci);

            % project the data down to one dimension, onto the vector
            % connecting the two centers
            v = fc(1,:,i) - fc(2,:,i);
            pdata = cdata * v'; pfc = fc(:,:,i) * v';
            mu = mean(pdata); sigma = std(pdata);

            % if sigma is too small in 1-d, then don't accept the split
            if (sigma < eps)
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                continue;
            end;

            pdata = (pdata - mu) ./ sigma;
            pfc = (pfc - mu) ./ sigma;


            lab = findlabels(pfc, pdata);

            % if there are too few points in the cluster, then don't accept the
            % split
            if ((sum(lab == 1) == 0) || (sum(lab == 2) == 0))
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                continue;
            end;

            keepSplit = 0;
            if (strcmp(testMethod, 'ad')) % Anderson-Darling statistical test
                AndersonStat(i) = andersondarling(pdata);
                if (AndersonStat(i) >= adcv)
                    keepSplit = 1;
                end;
            else % G-means statistical test based on the Gamma distribution
                dataC1 = pdata(lab == 1) - pfc(1);
                dataC2 = pdata(lab == 2) - pfc(2);

                r = sum(dataC1 .* dataC1) + sum(dataC2 .* dataC2);

                gamma = length(pdata) * (pi - 2) - 4;
                beta = 1/pi;
                prob = gamcdf(r, gamma, beta);

                if (prob < alpha)
                    keepSplit = 1;
                end;
            end;

            if (keepSplit == 1)
                newCtrs = [newCtrs; fc(:,:,i)];
                ind = [ind;i;i];
                newCtrIds = [newCtrIds; centerId + 1; centerId + 2];
                centerId = centerId + 2;
            else
                newCtrs = [newCtrs; c];
                ind = [ind;0];
                newCtrIds = [newCtrIds; centerIds(i)];
                rejectedSplits = union(rejectedSplits, centerIds(i));
            end;
        end;

        newK = size(newCtrs, 1);
        diff = newK-endK;
        if (diff > 0)
            [Val,Cands] = sort(AndersonStat);
            splitted = find(Val >= adcv);
            newCtrs(find(ismember(ind,Cands(splitted(1) : splitted(diff)))),:) = [];
            newCtrs = [newCtrs ; ctrs(Cands(splitted(1) : splitted(diff)),:)];
            ctrs = km(data, newCtrs);
            break
        elseif (diff == 0)
            ctrs = km(data, newCtrs);
            break
        end

        % run k-means on all centers and all data to update the fit
        ctrs = km(data, newCtrs);
        centerIds = newCtrIds;
    end;

    if (callback ~= 0)
        eval([callback, '(ctrs, data, ''finish'')']);
    end;

    centers = ctrs;

