function speech = S2S_synthesis_corr(p, file_list, data, corpus_dir, use_correlation)
% Really basic S2S synthesis module. Second version: OLA using Hanning
% window, with basic position correction using correlation. Partially
% inspired by both Yannis and Thierry's code for OLA and xcorr correction...
% 
% Usage: syn_speech = synthesize(p, file_list, data, corpus_dir, use_correlation)
% 
%    
% Parameters:
%    "p": parameter structure used in all the CC4M... stuff 
%         (used for p.fs, p.winlength, p.overlap)
%         IMPORTANT: a new field must be added to indicate the size of
%         the synthesis window:
%             p.synthesis_winlength
%    file_list: list of available files in the database (struture D:
%         D.name...) <-- comes from Yannis' code
%    data: matrix with two rows
%         data(1,:) : time positions of the centres of the windows for
%         synthesis (relative to the file in 2nd row)
%         data(2,:) : index of the file --> D(data(2,:)).name is the file name
%    corpus_dir: Base directory of the development audio files 
%
% Optional:
%    use_correlation: 0 or 1 (default if not present)
%         perform a correlation-based correction of the window position
%         prior to OLA process.

    p = CC4M_ParamInit(p);
    if nargin<5,
        use_correlation = 1;
    end
    
    analysis_winlength = floor(p.winlength * p.fs + 0.5);
    analysis_overlap = round(p.overlap * analysis_winlength);
    
    win_update_interval = analysis_winlength - analysis_overlap;
    
    synthesis_winlength = floor(p.synthesis_winlength * p.fs + 0.5);
    
    w_norm_factor = synthesis_winlength/2/win_update_interval;
    
    w = hanning(synthesis_winlength)/w_norm_factor;

    NUM_FRAMES = size(data, 2);
    
    speech = zeros((NUM_FRAMES-1)*win_update_interval + synthesis_winlength, 1);

    data(1,:)=data(1,:)/1e3; % Convert to seconds
    spe_end = 0;
    for m=1:NUM_FRAMES,

        file_name = strcat(corpus_dir, '/', file_list(data(2,m)).name);
        [y, Fs, N] = wavread(file_name);
        frame_centre = round(data(1,m) * Fs);

        if m>1 && use_correlation,
            modified_centre = get_modified_centre(sp_end, frame_centre);
            if (modified_centre ~= frame_centre)
                fprintf('Correction %d samples\n', modified_centre - ...
                        frame_centre);
            end
        else
            modified_centre = frame_centre;
        end
        
        frame = get_frame(modified_centre, synthesis_winlength);

        sp_beg = (m-1)*win_update_interval + 1;
        sp_end = sp_beg + synthesis_winlength - 1;
        
        speech(sp_beg:sp_end)=speech(sp_beg:sp_end)+w.*frame;
    end        

    skip_samples = (synthesis_winlength - analysis_winlength)/2;
    speech=speech(skip_samples+1:end-skip_samples);
    
    function  frame = get_frame(frame_centre, winlength)
        frame_beg = frame_centre - winlength/2 + 1;
        frame_end = frame_beg + winlength - 1;

        frame = subwave(frame_beg, frame_end);
    end
    
    function modified_centre = get_modified_centre(speech_length, frame_centre)
        
        buffer_size = synthesis_winlength - win_update_interval;
        if buffer_size > speech_length,
            buffer_size = speech_length;
        end    
        buffer = speech(speech_length-buffer_size+1:speech_length);
        
        %Un-window buffer to see if it helps in maximizing the correlation
        w2=w(end-buffer_size/2+1:end);
        buffer(end/2+1:end) = buffer(end/2+1:end)./w2;
        
        y_subwave_end = frame_centre + synthesis_winlength/2 - win_update_interval;
        y_subwave_beg = y_subwave_end - buffer_size + 1;    
        y_subwave = subwave(y_subwave_beg, y_subwave_end);    
        
        % Compute maximum of autocorrelation. Ideally this should result in
        %lag=0 (i.e. no phase mismatch).
        [correlation, lags] = xcorr(buffer, y_subwave, 'coef');
        [val, optimal_pos] = max(correlation);
        lag = lags(optimal_pos);
        
        modified_centre = frame_centre + lag;
    end

    function frame = subwave(frame_beg, frame_end)
    %Now it should be: y(frame_beg:frame_end), but let's do a sanity
    %check to ensure we're within limits
        frame_fill = [];
        if frame_beg < 1,
% $$$             fprintf('Frame %d, prepending %d 0s\n', m, -frame_beg+1);
            frame_fill = zeros(-frame_beg+1,1);
            frame_beg = 1;
        end
        frame_fill_end = [];
        if frame_end > length(y),
% $$$             fprintf('Frame %d, appending %d 0s\n', m, frame_end-length(y));
            frame_fill_end = zeros(frame_end-length(y),1);
            frame_end = length(y);
        end
        frame = [frame_fill; y(frame_beg:frame_end); frame_fill_end];
    end

end