#include <math.h>
#include "mex.h"

#define MAX_PREDICTOR_ORDER 100
#define EPSILON 1.0E-8

/************************************************************************/
/*									*/
/*	FUNCTION :	Del_Genin			    		*/
/*							    		*/
/*	PURPOSE  :	Solve the linear prediction problem 		*/
/*			with the Delsatres Genin algorithm  		*/
/*									*/
/*	ARGUMENTS  :							*/
/*		p -- Input parameter, moder order <= 30			*/
/*		r -- autocorrelation coefficients, dimension > p	*/
/*		k -- reflection coefficients, k[0] = 1, dimension > p 	*/
/*		a -- predictor  coefficients, a[0] = 1, dimension > p	*/
/*	RETURN VALUES :							*/
/*		k, a							*/
/*									*/
/* Stylianou Yannis , 1993						*/
/* email : styliano@sig.enst.fr						*/
/*									*/
/************************************************************************/

void Del_Genin(p, r, k, a)

int p;
double r[], k[], a[];
{

int	i, j, m;
double	lamda, f1, f2, temp,
	a_plus1[ MAX_PREDICTOR_ORDER / 2 + 1 ],
	a_plus2[ MAX_PREDICTOR_ORDER / 2 + 1 ];

	if( p > MAX_PREDICTOR_ORDER )
		mexErrMsgTxt("Order > MAX_PREDICTOR_ORDER (100) ");

	/*	Initialization	*/

k[0] = a[0] = 1;
 
if(r[0] < EPSILON )
    mexErrMsgTxt("r[0] < 1.0E-8");

k[1] = -r[1] / r[0];
if (fabs(k[1]) >= 1.0 )
   mexErrMsgTxt(" k[1] >= 1 ");

f1 = r[0] + r[1];
f2 = r[0] + r[2] + 2* r[1] * k[1];
lamda = f2 / f1;
k[2] = 1 - lamda / (1 + k[1] );

if (fabs(k[2]) >= 1.0 )
   mexErrMsgTxt(" k[2] >= 1 ");

a_plus1[0] = k[1] + k[1];
a_plus2[0] = -(r[1] + r[2] ) / f1;



	/*	Main loop	*/

for( m = 2; m< p; m++)
{
	j = m/2;
	f1 = f2;
	f2 = r[0] + r[m+1];
	for( i = 0; i < j; i++)
		f2 += (r[i+1] + r[m-i]) * a_plus2[i];
	if( m % 2)
		f2 += r[j+1] * a_plus2[j];

	lamda = f2 / f1;
	k[m+1] = 1 - lamda / (1 + k[m]);

	if (fabs(k[m+1]) >= 1.0 )
		{
		printf("%d\n", m);
   		mexErrMsgTxt(" k[m+1] >= 1 ");
		}

	for( i = 0; i < j; i++)
	{
	temp		= a_plus2[i];
	a_plus2[i] 	= a_plus1[i];
	a_plus1[i]	= temp;
	}

	if( m % 2 )
		a_plus1[j] = a_plus2[j];

	i = j;
	if( m % 2 == 0 )
	{
	i--;
	a_plus2[i+1] = a_plus1[i] + a_plus1[i] - lamda * a_plus2[i];
	}

	for( ; i > 0; i-- )
	   a_plus2[i] = a_plus1[i] + a_plus1[i-1] - lamda * a_plus2[i-1];
	a_plus2[0] = a_plus1[0] + 1 - lamda;
}

/* Computation of predictor coefficients */

a[1] = a_plus2[0] - k[p];
for( m = 2, i = 1, j = 0; m <= (p + 1) / 2; m++, i++, j++ )
   a[m] = a[m-1] + a_plus2[i] - (1+k[p] ) * a_plus1[j];
if( p % 2 )
 {
 i--;
 j--;
 }

for( i--; m < p; m++, i--, j--)
 a[m] = a[m-1] + a_plus2[i] - (1+k[p]) * a_plus1[j];
a[p] = k[p];

}	

#ifdef __STDC__
void mexFunction(int nlhs, Matrix *plhs[], int nrhs, Matrix *prhs[])
#else
mexFunction(nlhs, plhs, nrhs, prhs)
int nlhs;
Matrix *plhs[];
int nrhs;
Matrix *prhs[];
#endif
{

double *r, *k, *a;
int p;

p = (int)mxGetScalar(prhs[1]);


plhs[0] = mxCreateFull(1,p+1,REAL);
plhs[1] = mxCreateFull(1,p+1,REAL);

a = mxGetPr(plhs[0]);
k = mxGetPr(plhs[1]);

r = mxGetPr(prhs[0]);

Del_Genin(p, r, k, a);
}
